# Android Open Accessory Protocol 2.0 #

This document describes the changes to the Android Open Accessory (AOA) protocol since its
initial release, and is a supplement to the documentation of the
[first release of AOA](/tech/accessories/aoap/aoa.html).

The Android Open Accessory Protocol 2.0 adds two new features: audio output (from the Android
device to the accessory) and support for the accessory acting as one or more Human Interface Devices
(HID) to the Android device. The Android SDK APIs available to Android application developers
remain unchanged.


## Detecting Android Open Accessory 2.0 Support ##

In order for an accessory to determine if a connected Android device supports accessories and at
what protocol level, the accessory must send a `getProtocol()` command and check the result.
Android devices supporting the initial version of the Android Open Accessory protocol return a
`1`, representing the protocol version number. Devices that support the new features described
in this document must return `2` for the protocol version. Version 2.0 of the protocol is
upwardly compatible, so accessories designed for the original accessory protocol still work
with newer Android devices. The following example from the Accessory Development Kit 2011
[source code](http://developer.android.com/tools/adk/adk2.html#src-download)
(`<adk-src>/adk1/board/AndroidAccessory/AndroidAccessory.cpp`) library demonstrates this protocol
check:

    bool AndroidAccessory::switchDevice(byte addr)
    {
        int protocol = getProtocol(addr);
        if (protocol >= 1) {
            Serial.print("device supports protocol 1 or higher\n");
        } else {
            Serial.print("could not read device protocol version\n");
            return false;
        }

        sendString(addr, ACCESSORY_STRING_MANUFACTURER, manufacturer);
        sendString(addr, ACCESSORY_STRING_MODEL, model);
        sendString(addr, ACCESSORY_STRING_DESCRIPTION, description);
        sendString(addr, ACCESSORY_STRING_VERSION, version);
        sendString(addr, ACCESSORY_STRING_URI, uri);
        sendString(addr, ACCESSORY_STRING_SERIAL, serial);

        usb.ctrlReq(addr, 0, USB_SETUP_HOST_TO_DEVICE | USB_SETUP_TYPE_VENDOR |
                    USB_SETUP_RECIPIENT_DEVICE, ACCESSORY_START, 0, 0, 0, 0, NULL);
        return true;
    }

AOA 2.0 includes new USB product IDs, one for each combination of USB interfaces available when
in accessory mode. The possible USB interfaces are:

- **accessory** - An interface providing 2 bulk endpoints for communicating with an
Android application.
- **audio** - A new standard USB audio class interface for streaming audio
from an Android device to an accessory.
- **adb** - An interface intended only for debugging purposes while developing an
accessory. Only enabled if the user has USB Debugging enabled in Settings on the Android device.


In AOA 1.0, there are only two USB product IDs:

- `0x2D00` - accessory
- `0x2D01` - accessory + adb

AOA 2.0 adds an optional USB audio interface and, therefore, includes product IDs for the new
combinations of USB interfaces:

- `0x2D02` - audio
- `0x2D03` - audio + adb
- `0x2D04` - accessory + audio
- `0x2D05` - accessory + audio + adb


## Audio Support ##

AOA 2.0 includes optional support for audio output from an Android device to an accessory. This
version of the protocol supports a standard USB audio class interface that is capable of 2 channel
16-bit PCM audio with a bit rate of 44100 Khz. AOA 2.0 is currently limited to this output mode, but
additional audio modes may be added in the future.

To enable the audio support, the accessory must send a new USB control request:

    **SET_AUDIO_MODE**
    requestType:    USB_DIR_OUT | USB_TYPE_VENDOR
    request:        58
    value:          0 for no audio (default),
                    1 for 2 channel, 16-bit PCM at 44100 KHz
    index:          0
    data            none

This command must be sent _before_ sending the `ACCESSORY_START` command for
entering accessory mode.


## HID Support ##

AOA 2.0 allows the accessory to register one or more USB Human Interface Devices (HID) with
an Android device. This approach reverses the direction of communication for typical USB HID
devices like USB mice and keyboards. Normally, the HID device is a peripheral connected to a USB
host like a personal computer. But in the case of the AOA protocol, the USB host acts as one or more
input devices to a USB peripheral.

HID support in AOA 2.0 is simply a proxy for standard HID events. The implementation makes no
assumptions about the content or type of events and merely passes it through to the input system,
so an AOA 2.0 accessory can act as any HID device (mouse, keyboard, game controller, etc.). It
can be used for something as simple as the play/pause button on a media dock, or something as
complicated as a docking station with a mouse and full QWERTY keyboard.

The AOA 2.0 protocol adds four new USB control requests to allow the accessory to act as one or
more HID input devices to the Android device.  Since HID support is done entirely through
control requests on endpoint zero, no new USB interface is needed to provide this support. The
control requests are as follows:

- **ACCESSORY_REGISTER_HID** registers a new HID device with the Android device.
The accessory provides an ID number that is used to identify the HID device for the other three
calls. This ID is valid until USB is disconnected or until the accessory sends
`ACCESSORY_UNREGISTER_HID` to unregister the HID device.
- **ACCESSORY_UNREGISTER_HID** unregisters a HID device that was previously
registered with `ACCESSORY_REGISTER_HID`.
- **ACCESSORY_SET_HID_REPORT_DESC** sends a report descriptor for a HID device to
the Android device. This request is used to describe the capabilities of the HID device, and must
be sent before reporting any HID events to the Android device. If the report descriptor is larger
than the maximum packet size for endpoint zero, multiple `ACCESSORY_SET_HID_REPORT_DESC` commands
are sent in order to transfer the entire descriptor.
- **ACCESSORY_SEND_HID_EVENT** sends input events from the accessory to the Android
device.

The code definitions for these new control requests are as follows:

    /* Control request for registering a HID device.
     * Upon registering, a unique ID is sent by the accessory in the
     * value parameter. This ID will be used for future commands for
     * the device
     *
     *  requestType:    USB_DIR_OUT | USB_TYPE_VENDOR
     *  request:        ACCESSORY_REGISTER_HID_DEVICE
     *  value:          Accessory assigned ID for the HID device
     *  index:          total length of the HID report descriptor
     *  data            none
     */
    #define ACCESSORY_REGISTER_HID         54

    /* Control request for unregistering a HID device.
     *
     *  requestType:    USB_DIR_OUT | USB_TYPE_VENDOR
     *  request:        ACCESSORY_REGISTER_HID
     *  value:          Accessory assigned ID for the HID device
     *  index:          0
     *  data            none
     */
    #define ACCESSORY_UNREGISTER_HID         55

    /* Control request for sending the HID report descriptor.
     * If the HID descriptor is longer than the endpoint zero max packet size,
     * the descriptor will be sent in multiple ACCESSORY_SET_HID_REPORT_DESC
     * commands. The data for the descriptor must be sent sequentially
     * if multiple packets are needed.
     *
     *  requestType:    USB_DIR_OUT | USB_TYPE_VENDOR
     *  request:        ACCESSORY_SET_HID_REPORT_DESC
     *  value:          Accessory assigned ID for the HID device
     *  index:          offset of data in descriptor
     *                      (needed when HID descriptor is too big for one packet)
     *  data            the HID report descriptor
     */
    #define ACCESSORY_SET_HID_REPORT_DESC         56

    /* Control request for sending HID events.
     *
     *  requestType:    USB_DIR_OUT | USB_TYPE_VENDOR
     *  request:        ACCESSORY_SEND_HID_EVENT
     *  value:          Accessory assigned ID for the HID device
     *  index:          0
     *  data            the HID report for the event
     */
    #define ACCESSORY_SEND_HID_EVENT         57


## Interoperability with AOA 1.0 Features ##

The original [AOA protocol](/tech/accessories/aoap/aoa.html) provided support for an Android
application to communicate directly with a USB host (accessory) over USB. AOA 2.0 keeps that
support, but adds new features to allow the accessory to communicate with the Android operating
system itself (specifically the audio and input systems). The design of the AOA 2.0 makes it is
possible to build an accessory that also makes use of the new audio and/or HID support in addition
to the original feature set. Simply use the new features described in this document in addition to
the original AOA protocol features.


## Connecting AOA 2.0 without an Android App ##

It is possible to design an accessory (for example, an audio dock) that uses the new audio and
HID support, but does not need to communicate with an application on the Android device. In that
case, the user would not want to see the dialog prompts related to finding and associating the newly
attached accessory with an Android application that can communicate with it. To prevent these
dialogs from appearing after the device and accessory are connected, the accessory can simply not
send the manufacturer and model names to the Android device. If these strings are not provided to
the Android device, then the accessory is able to make use of the new audio and HID support in AOA
2.0 without the system attempting to find an application to communicate with the accessory. Also,
if these strings are not provided, the accessory USB interface is not present in the Android
device USB configuration after the device enters accessory mode.