/*
 * Copyright 2000-2012 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.util.xml.stubs;

import com.intellij.openapi.vfs.VirtualFileFilter;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiFile;
import com.intellij.psi.impl.PsiManagerImpl;
import com.intellij.psi.xml.XmlAttribute;
import com.intellij.psi.xml.XmlFile;
import com.intellij.psi.xml.XmlTag;
import com.intellij.util.xml.*;
import com.intellij.util.xml.stubs.model.Bar;
import com.intellij.util.xml.stubs.model.Foo;
import com.intellij.util.xml.stubs.model.NotStubbed;

import java.util.List;

/**
 * @author Dmitry Avdeev
 *         Date: 8/8/12
 */
public class DomStubUsingTest extends DomStubTest {

  public void testFoo() throws Exception {

    DomFileElement<Foo> fileElement = prepare("foo.xml", Foo.class);
    PsiFile file = fileElement.getFile();
    assertFalse(file.getNode().isParsed());

    Foo foo = fileElement.getRootElement();
    List<Bar> bars = foo.getBars();
    assertFalse(file.getNode().isParsed());

    assertEquals(2, bars.size());
    Bar bar = bars.get(0);
    String value = bar.getString().getStringValue();
    assertEquals("xxx", value);

    Object o = bar.getString().getValue();
    assertEquals("xxx", o);

    Integer integer = bar.getInt().getValue();
    assertEquals(666, integer.intValue());

    assertFalse(file.getNode().isParsed());

    Bar emptyBar = bars.get(1);
    GenericAttributeValue<String> string = emptyBar.getString();
    assertNull(string.getXmlElement());

    assertFalse(file.getNode().isParsed());
  }

  public void testAccessingPsi() throws Exception {
    DomFileElement<Foo> element = prepare("foo.xml", Foo.class);
    assertNotNull(element.getXmlElement());

    XmlTag tag = element.getRootTag();
    assertNotNull(tag);

    Foo foo = element.getRootElement();
    assertNotNull(foo.getXmlTag());

    Bar bar = foo.getBars().get(0);
    assertNotNull(bar.getXmlElement());

    XmlAttribute attribute = bar.getString().getXmlAttribute();
    assertNotNull(attribute);
  }

  public void testConverters() throws Exception {
    DomFileElement<Foo> element = prepare("converters.xml", Foo.class);
    Bar bar = element.getRootElement().getBars().get(0);
    PsiClass value = bar.getClazz().getValue();
    assertNotNull(value);
    assertEquals("java.lang.String", value.getQualifiedName());
    assertFalse(element.getFile().getNode().isParsed());
  }

  public void testParent() throws Exception {
    DomFileElement<Foo> element = prepare("parent.xml", Foo.class);

    Bar bar = element.getRootElement().getBars().get(0);
    GenericAttributeValue<Integer> notStubbed = bar.getNotStubbed();
    DomElement parent = notStubbed.getParent();
    assertEquals(bar, parent);

    NotStubbed child = bar.getNotStubbeds().get(0);
    parent = child.getParent();
    assertEquals(bar, parent);
  }

  public void testChildrenOfType() throws Exception {

    DomFileElement<Foo> element = prepare("foo.xml", Foo.class);
    Foo foo = element.getRootElement();
    List<Bar> bars = DomUtil.getChildrenOf(foo, Bar.class);
    assertEquals(2, bars.size());
  }

  public void testFileLoading() throws Exception {
    XmlFile file = prepareFile("foo.xml");
    ((PsiManagerImpl)getPsiManager()).setAssertOnFileLoadingFilter(VirtualFileFilter.ALL);
    try {
      DomManager.getDomManager(getProject()).getFileElement(file, Foo.class);
    }
    finally {
      ((PsiManagerImpl)getPsiManager()).setAssertOnFileLoadingFilter(VirtualFileFilter.NONE);
    }
  }
}
