//===--- StringRef.h - Constant String Reference Wrapper --------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_ADT_STRINGREF_H
#define LLVM_ADT_STRINGREF_H

#include <algorithm>
#include <cassert>
#include <cstring>
#include <string>

namespace llvm {

  /// StringRef - Represent a constant reference to a string, i.e. a character
  /// array and a length, which need not be null terminated.
  ///
  /// This class does not own the string data, it is expected to be used in
  /// situations where the character data resides in some other buffer, whose
  /// lifetime extends past that of the StringRef. For this reason, it is not in
  /// general safe to store a StringRef.
  class StringRef {
  public:
    typedef const char *iterator;
    static const size_t npos = ~size_t(0);
    typedef size_t size_type;

  private:
    /// The start of the string, in an external buffer.
    const char *Data;

    /// The length of the string.
    size_t Length;

  public:
    /// @name Constructors
    /// @{

    /// Construct an empty string ref.
    /*implicit*/ StringRef() : Data(0), Length(0) {}

    /// Construct a string ref from a cstring.
    /*implicit*/ StringRef(const char *Str)
      : Data(Str) { if (Str) Length = ::strlen(Str); else Length = 0; }

    /// Construct a string ref from a pointer and length.
    /*implicit*/ StringRef(const char *data, size_t length)
      : Data(data), Length(length) {}

    /// Construct a string ref from an std::string.
    /*implicit*/ StringRef(const std::string &Str)
      : Data(Str.c_str()), Length(Str.length()) {}

    /// @}
    /// @name Iterators
    /// @{

    iterator begin() const { return Data; }

    iterator end() const { return Data + Length; }

    /// @}
    /// @name String Operations
    /// @{

    /// data - Get a pointer to the start of the string (which may not be null
    /// terminated).
    const char *data() const { return Data; }

    /// empty - Check if the string is empty.
    bool empty() const { return Length == 0; }

    /// size - Get the string size.
    size_t size() const { return Length; }

    /// front - Get the first character in the string.
    char front() const {
      assert(!empty());
      return Data[0];
    }

    /// back - Get the last character in the string.
    char back() const {
      assert(!empty());
      return Data[Length-1];
    }

    /// equals - Check for string equality, this is more efficient than
    /// compare() when the relative ordering of inequal strings isn't needed.
    bool equals(const StringRef &RHS) const {
      return (Length == RHS.Length &&
              memcmp(Data, RHS.Data, RHS.Length) == 0);
    }

    /// compare - Compare two strings; the result is -1, 0, or 1 if this string
    /// is lexicographically less than, equal to, or greater than the \arg RHS.
    int compare(const StringRef &RHS) const {
      // Check the prefix for a mismatch.
      if (int Res = memcmp(Data, RHS.Data, std::min(Length, RHS.Length)))
        return Res < 0 ? -1 : 1;

      // Otherwise the prefixes match, so we only need to check the lengths.
      if (Length == RHS.Length)
        return 0;
      return Length < RHS.Length ? -1 : 1;
    }

    /// str - Get the contents as an std::string.
    std::string str() const { return std::string(Data, Length); }

    /// @}
    /// @name Operator Overloads
    /// @{

    char operator[](size_t Index) const {
      assert(Index < Length && "Invalid index!");
      return Data[Index];
    }

    /// @}
    /// @name Type Conversions
    /// @{

    operator std::string() const {
      return str();
    }

    /// @}
    /// @name String Predicates
    /// @{

    /// startswith - Check if this string starts with the given \arg Prefix.
    bool startswith(const StringRef &Prefix) const {
      return substr(0, Prefix.Length).equals(Prefix);
    }

    /// endswith - Check if this string ends with the given \arg Suffix.
    bool endswith(const StringRef &Suffix) const {
      return slice(size() - Suffix.Length, size()).equals(Suffix);
    }

    /// @}
    /// @name String Searching
    /// @{

    /// find - Search for the first character \arg C in the string.
    ///
    /// \return - The index of the first occurence of \arg C, or npos if not
    /// found.
    size_t find(char C) const {
      for (size_t i = 0, e = Length; i != e; ++i)
        if (Data[i] == C)
          return i;
      return npos;
    }

    /// find - Search for the first string \arg Str in the string.
    ///
    /// \return - The index of the first occurence of \arg Str, or npos if not
    /// found.
    size_t find(const StringRef &Str) const;

    /// rfind - Search for the last character \arg C in the string.
    ///
    /// \return - The index of the last occurence of \arg C, or npos if not
    /// found.
    size_t rfind(char C, size_t From = npos) const {
      From = std::min(From, Length);
      size_t i = From;
      while (i != 0) {
        --i;
        if (Data[i] == C)
          return i;
      }
      return npos;
    }

    /// rfind - Search for the last string \arg Str in the string.
    ///
    /// \return - The index of the last occurence of \arg Str, or npos if not
    /// found.
    size_t rfind(const StringRef &Str) const;

    /// find_first_of - Find the first instance of the specified character or
    /// return npos if not in string.  Same as find.
    size_type find_first_of(char C) const { return find(C); }

    /// find_first_of - Find the first character from the string 'Chars' in the
    /// current string or return npos if not in string.
    size_type find_first_of(StringRef Chars) const;

    /// find_first_not_of - Find the first character in the string that is not
    /// in the string 'Chars' or return npos if all are in string. Same as find.
    size_type find_first_not_of(StringRef Chars) const;

    /// @}
    /// @name Helpful Algorithms
    /// @{

    /// count - Return the number of occurrences of \arg C in the string.
    size_t count(char C) const {
      size_t Count = 0;
      for (size_t i = 0, e = Length; i != e; ++i)
        if (Data[i] == C)
          ++Count;
      return Count;
    }

    /// count - Return the number of non-overlapped occurrences of \arg Str in
    /// the string.
    size_t count(const StringRef &Str) const;

    /// getAsInteger - Parse the current string as an integer of the specified
    /// radix.  If Radix is specified as zero, this does radix autosensing using
    /// extended C rules: 0 is octal, 0x is hex, 0b is binary.
    ///
    /// If the string is invalid or if only a subset of the string is valid,
    /// this returns true to signify the error.  The string is considered
    /// erroneous if empty.
    ///
    bool getAsInteger(unsigned Radix, long long &Result) const;
    bool getAsInteger(unsigned Radix, unsigned long long &Result) const;
    bool getAsInteger(unsigned Radix, int &Result) const;
    bool getAsInteger(unsigned Radix, unsigned &Result) const;

    // TODO: Provide overloads for int/unsigned that check for overflow.

    /// @}
    /// @name Substring Operations
    /// @{

    /// substr - Return a reference to the substring from [Start, Start + N).
    ///
    /// \param Start - The index of the starting character in the substring; if
    /// the index is npos or greater than the length of the string then the
    /// empty substring will be returned.
    ///
    /// \param N - The number of characters to included in the substring. If N
    /// exceeds the number of characters remaining in the string, the string
    /// suffix (starting with \arg Start) will be returned.
    StringRef substr(size_t Start, size_t N = npos) const {
      Start = std::min(Start, Length);
      return StringRef(Data + Start, std::min(N, Length - Start));
    }

    /// slice - Return a reference to the substring from [Start, End).
    ///
    /// \param Start - The index of the starting character in the substring; if
    /// the index is npos or greater than the length of the string then the
    /// empty substring will be returned.
    ///
    /// \param End - The index following the last character to include in the
    /// substring. If this is npos, or less than \arg Start, or exceeds the
    /// number of characters remaining in the string, the string suffix
    /// (starting with \arg Start) will be returned.
    StringRef slice(size_t Start, size_t End) const {
      Start = std::min(Start, Length);
      End = std::min(std::max(Start, End), Length);
      return StringRef(Data + Start, End - Start);
    }

    /// split - Split into two substrings around the first occurence of a
    /// separator character.
    ///
    /// If \arg Separator is in the string, then the result is a pair (LHS, RHS)
    /// such that (*this == LHS + Separator + RHS) is true and RHS is
    /// maximal. If \arg Separator is not in the string, then the result is a
    /// pair (LHS, RHS) where (*this == LHS) and (RHS == "").
    ///
    /// \param Separator - The character to split on.
    /// \return - The split substrings.
    std::pair<StringRef, StringRef> split(char Separator) const {
      size_t Idx = find(Separator);
      if (Idx == npos)
        return std::make_pair(*this, StringRef());
      return std::make_pair(slice(0, Idx), slice(Idx+1, npos));
    }

    /// rsplit - Split into two substrings around the last occurence of a
    /// separator character.
    ///
    /// If \arg Separator is in the string, then the result is a pair (LHS, RHS)
    /// such that (*this == LHS + Separator + RHS) is true and RHS is
    /// minimal. If \arg Separator is not in the string, then the result is a
    /// pair (LHS, RHS) where (*this == LHS) and (RHS == "").
    ///
    /// \param Separator - The character to split on.
    /// \return - The split substrings.
    std::pair<StringRef, StringRef> rsplit(char Separator) const {
      size_t Idx = rfind(Separator);
      if (Idx == npos)
        return std::make_pair(*this, StringRef());
      return std::make_pair(slice(0, Idx), slice(Idx+1, npos));
    }

    /// @}
  };

  /// @name StringRef Comparison Operators
  /// @{

  inline bool operator==(const StringRef &LHS, const StringRef &RHS) {
    return LHS.equals(RHS);
  }

  inline bool operator!=(const StringRef &LHS, const StringRef &RHS) {
    return !(LHS == RHS);
  }

  inline bool operator<(const StringRef &LHS, const StringRef &RHS) {
    return LHS.compare(RHS) == -1;
  }

  inline bool operator<=(const StringRef &LHS, const StringRef &RHS) {
    return LHS.compare(RHS) != 1;
  }

  inline bool operator>(const StringRef &LHS, const StringRef &RHS) {
    return LHS.compare(RHS) == 1;
  }

  inline bool operator>=(const StringRef &LHS, const StringRef &RHS) {
    return LHS.compare(RHS) != -1;
  }

  /// @}

}

#endif
