//===- llvm/Analysis/MemoryBuiltins.h- Calls to memory builtins -*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
//
// This family of functions identifies calls to builtin functions that allocate
// or free memory.  
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_ANALYSIS_MEMORYBUILTINS_H
#define LLVM_ANALYSIS_MEMORYBUILTINS_H

namespace llvm {
class CallInst;
class LLVMContext;
class PointerType;
class TargetData;
class Type;
class Value;

//===----------------------------------------------------------------------===//
//  malloc Call Utility Functions.
//

/// isMalloc - Returns true if the value is either a malloc call or a bitcast of 
/// the result of a malloc call
bool isMalloc(const Value* I);

/// extractMallocCall - Returns the corresponding CallInst if the instruction
/// is a malloc call.  Since CallInst::CreateMalloc() only creates calls, we
/// ignore InvokeInst here.
const CallInst* extractMallocCall(const Value* I);
CallInst* extractMallocCall(Value* I);

/// extractMallocCallFromBitCast - Returns the corresponding CallInst if the
/// instruction is a bitcast of the result of a malloc call.
const CallInst* extractMallocCallFromBitCast(const Value* I);
CallInst* extractMallocCallFromBitCast(Value* I);

/// isArrayMalloc - Returns the corresponding CallInst if the instruction 
/// matches the malloc call IR generated by CallInst::CreateMalloc().  This 
/// means that it is a malloc call with one bitcast use AND the malloc call's 
/// size argument is:
///  1. a constant not equal to the size of the malloced type
/// or
///  2. the result of a multiplication by the size of the malloced type
/// Otherwise it returns NULL.
/// The unique bitcast is needed to determine the type/size of the array
/// allocation.
CallInst* isArrayMalloc(Value* I, LLVMContext &Context, const TargetData* TD);
const CallInst* isArrayMalloc(const Value* I, LLVMContext &Context,
                              const TargetData* TD);

/// getMallocType - Returns the PointerType resulting from the malloc call.
/// This PointerType is the result type of the call's only bitcast use.
/// If there is no unique bitcast use, then return NULL.
const PointerType* getMallocType(const CallInst* CI);

/// getMallocAllocatedType - Returns the Type allocated by malloc call. This
/// Type is the result type of the call's only bitcast use. If there is no
/// unique bitcast use, then return NULL.
const Type* getMallocAllocatedType(const CallInst* CI);

/// getMallocArraySize - Returns the array size of a malloc call.  For array
/// mallocs, the size is computated in 1 of 3 ways:
///  1. If the element type is of size 1, then array size is the argument to 
///     malloc.
///  2. Else if the malloc's argument is a constant, the array size is that
///     argument divided by the element type's size.
///  3. Else the malloc argument must be a multiplication and the array size is
///     the first operand of the multiplication.
/// For non-array mallocs, the computed size is constant 1. 
/// This function returns NULL for all mallocs whose array size cannot be
/// determined.
Value* getMallocArraySize(CallInst* CI, LLVMContext &Context,
                          const TargetData* TD);
                          
//===----------------------------------------------------------------------===//
//  free Call Utility Functions.
//

/// isFreeCall - Returns true if the the value is a call to the builtin free()
bool isFreeCall(const Value* I);

} // End llvm namespace

#endif
