//===--- Type.h - C Language Family Type Representation ---------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
//
//  This file defines the Type interface and subclasses.
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_CLANG_AST_TYPE_H
#define LLVM_CLANG_AST_TYPE_H

#include "llvm/Support/Casting.h"
#include "llvm/ADT/FoldingSet.h"
#include "llvm/ADT/APSInt.h"
#include "llvm/Bitcode/SerializationFwd.h"

using llvm::isa;
using llvm::cast;
using llvm::cast_or_null;
using llvm::dyn_cast;
using llvm::dyn_cast_or_null;

namespace clang {
  class ASTContext;
  class Type;
  class TypedefDecl;
  class TagDecl;
  class RecordDecl;
  class EnumDecl;
  class FieldDecl;
  class ObjCInterfaceDecl;
  class ObjCProtocolDecl;
  class ObjCMethodDecl;
  class Expr;
  class SourceLocation;
  class PointerLikeType;
  class PointerType;
  class ReferenceType;
  class VectorType;
  class ArrayType;
  class ConstantArrayType;
  class VariableArrayType;
  class IncompleteArrayType;
  class RecordType;
  class ComplexType;
  class TagType;
  class FunctionType;
  class OCUVectorType;
  class BuiltinType;
  class ObjCInterfaceType;
  class ObjCQualifiedIdType;
  class ObjCQualifiedInterfaceType;
  class StmtIteratorBase;
  
/// QualType - For efficiency, we don't store CVR-qualified types as nodes on
/// their own: instead each reference to a type stores the qualifiers.  This
/// greatly reduces the number of nodes we need to allocate for types (for
/// example we only need one for 'int', 'const int', 'volatile int',
/// 'const volatile int', etc).
///
/// As an added efficiency bonus, instead of making this a pair, we just store
/// the three bits we care about in the low bits of the pointer.  To handle the
/// packing/unpacking, we make QualType be a simple wrapper class that acts like
/// a smart pointer.
class QualType {
  uintptr_t ThePtr;
public:
  enum TQ {   // NOTE: These flags must be kept in sync with DeclSpec::TQ.
    Const    = 0x1,
    Restrict = 0x2,
    Volatile = 0x4,
    CVRFlags = Const|Restrict|Volatile
  };
  
  QualType() : ThePtr(0) {}
  
  QualType(Type *Ptr, unsigned Quals) {
    assert((Quals & ~CVRFlags) == 0 && "Invalid type qualifiers!");
    ThePtr = reinterpret_cast<uintptr_t>(Ptr);
    assert((ThePtr & CVRFlags) == 0 && "Type pointer not 8-byte aligned?");
    ThePtr |= Quals;
  }

  static QualType getFromOpaquePtr(void *Ptr) {
    QualType T;
    T.ThePtr = reinterpret_cast<uintptr_t>(Ptr);
    return T;
  }
  
  unsigned getCVRQualifiers() const {
    return ThePtr & CVRFlags;
  }
  Type *getTypePtr() const {
    return reinterpret_cast<Type*>(ThePtr & ~CVRFlags);
  }
  
  void *getAsOpaquePtr() const {
    return reinterpret_cast<void*>(ThePtr);
  }
  
  Type &operator*() const {
    return *getTypePtr();
  }

  Type *operator->() const {
    return getTypePtr();
  }
  
  /// isNull - Return true if this QualType doesn't point to a type yet.
  bool isNull() const {
    return ThePtr == 0;
  }

  bool isConstQualified() const {
    return (ThePtr & Const) ? true : false;
  }
  bool isVolatileQualified() const {
    return (ThePtr & Volatile) ? true : false;
  }
  bool isRestrictQualified() const {
    return (ThePtr & Restrict) ? true : false;
  }
  
  /// addConst/addVolatile/addRestrict - add the specified type qual to this
  /// QualType.
  void addConst()    { ThePtr |= Const; }
  void addVolatile() { ThePtr |= Volatile; }
  void addRestrict() { ThePtr |= Restrict; }

  QualType getQualifiedType(unsigned TQs) const {
    return QualType(getTypePtr(), TQs);
  }
  
  inline QualType getUnqualifiedType() const;
  
  /// operator==/!= - Indicate whether the specified types and qualifiers are
  /// identical.
  bool operator==(const QualType &RHS) const {
    return ThePtr == RHS.ThePtr;
  }
  bool operator!=(const QualType &RHS) const {
    return ThePtr != RHS.ThePtr;
  }
  std::string getAsString() const {
    std::string S;
    getAsStringInternal(S);
    return S;
  }
  void getAsStringInternal(std::string &Str) const;
  
  void dump(const char *s = 0) const;

//private:
  /// getCanonicalType - Return the canonical version of this type, with the
  /// appropriate type qualifiers on it.
  inline QualType getCanonicalType() const;
public:
  
  /// getAddressSpace - Return the address space of this type.
  inline unsigned getAddressSpace() const;
  
  /// Emit - Serialize a QualType to Bitcode.
  void Emit(llvm::Serializer& S) const;
  
  /// Read - Deserialize a QualType from Bitcode.
  static QualType ReadVal(llvm::Deserializer& D);
  
private:
  void ReadBackpatch(llvm::Deserializer& D);
  friend class FieldDecl;
};

} // end clang.

namespace llvm {
/// Implement simplify_type for QualType, so that we can dyn_cast from QualType
/// to a specific Type class.
template<> struct simplify_type<const ::clang::QualType> {
  typedef ::clang::Type* SimpleType;
  static SimpleType getSimplifiedValue(const ::clang::QualType &Val) {
    return Val.getTypePtr();
  }
};
template<> struct simplify_type< ::clang::QualType>
  : public simplify_type<const ::clang::QualType> {};
  
} // end namespace llvm

namespace clang {

/// Type - This is the base class of the type hierarchy.  A central concept
/// with types is that each type always has a canonical type.  A canonical type
/// is the type with any typedef names stripped out of it or the types it
/// references.  For example, consider:
///
///  typedef int  foo;
///  typedef foo* bar;
///    'int *'    'foo *'    'bar'
///
/// There will be a Type object created for 'int'.  Since int is canonical, its
/// canonicaltype pointer points to itself.  There is also a Type for 'foo' (a
/// TypeNameType).  Its CanonicalType pointer points to the 'int' Type.  Next
/// there is a PointerType that represents 'int*', which, like 'int', is
/// canonical.  Finally, there is a PointerType type for 'foo*' whose canonical
/// type is 'int*', and there is a TypeNameType for 'bar', whose canonical type
/// is also 'int*'.
///
/// Non-canonical types are useful for emitting diagnostics, without losing
/// information about typedefs being used.  Canonical types are useful for type
/// comparisons (they allow by-pointer equality tests) and useful for reasoning
/// about whether something has a particular form (e.g. is a function type),
/// because they implicitly, recursively, strip all typedefs out of a type.
///
/// Types, once created, are immutable.
///
class Type {
public:
  enum TypeClass {
    Builtin, Complex, Pointer, Reference, 
    ConstantArray, VariableArray, IncompleteArray,
    Vector, OCUVector,
    FunctionNoProto, FunctionProto,
    TypeName, Tagged, ASQual,
    ObjCInterface, ObjCQualifiedInterface,
    ObjCQualifiedId,
    TypeOfExp, TypeOfTyp // GNU typeof extension.
  };
private:
  QualType CanonicalType;

  /// TypeClass bitfield - Enum that specifies what subclass this belongs to.
  /// Note that this should stay at the end of the ivars for Type so that
  /// subclasses can pack their bitfields into the same word.
  unsigned TC : 5;
protected:
  // silence VC++ warning C4355: 'this' : used in base member initializer list
  Type *this_() { return this; }
  Type(TypeClass tc, QualType Canonical)
    : CanonicalType(Canonical.isNull() ? QualType(this_(), 0) : Canonical),
      TC(tc) {}
  virtual ~Type();
  friend class ASTContext;
  
  void EmitTypeInternal(llvm::Serializer& S) const;
  void ReadTypeInternal(llvm::Deserializer& D);
  
public:
  TypeClass getTypeClass() const { return static_cast<TypeClass>(TC); }
  
  bool isCanonical() const { return CanonicalType.getTypePtr() == this; }

  /// Types are partitioned into 3 broad categories (C99 6.2.5p1): 
  /// object types, function types, and incomplete types.
  
  /// isObjectType - types that fully describe objects. An object is a region
  /// of memory that can be examined and stored into (H&S).
  bool isObjectType() const;

  /// isIncompleteType - Return true if this is an incomplete type.
  /// A type that can describe objects, but which lacks information needed to
  /// determine its size (e.g. void, or a fwd declared struct). Clients of this
  /// routine will need to determine if the size is actually required.  
  bool isIncompleteType() const;

  /// isIncompleteOrObjectType - Return true if this is an incomplete or object
  /// type, in other words, not a function type.
  bool isIncompleteOrObjectType() const {
    return !isFunctionType();
  }
  
  /// isVariablyModifiedType (C99 6.7.5.2p2) - Return true for variable array
  /// types that have a non-constant expression. This does not include "[]".
  bool isVariablyModifiedType() const;
  
  /// isIncompleteArrayType (C99 6.2.5p22) - Return true for variable array
  /// types that don't have any expression ("[]").
  bool isIncompleteArrayType() const;
  
  /// Helper methods to distinguish type categories. All type predicates
  /// operate on the canonical type, ignoring typedefs and qualifiers.
  
  /// isIntegerType() does *not* include complex integers (a GCC extension).
  /// isComplexIntegerType() can be used to test for complex integers.
  bool isIntegerType() const;     // C99 6.2.5p17 (int, char, bool, enum)
  bool isEnumeralType() const;
  bool isBooleanType() const;
  bool isCharType() const;
  bool isIntegralType() const;
  
  /// Floating point categories.
  bool isRealFloatingType() const; // C99 6.2.5p10 (float, double, long double)
  /// isComplexType() does *not* include complex integers (a GCC extension).
  /// isComplexIntegerType() can be used to test for complex integers.
  bool isComplexType() const;      // C99 6.2.5p11 (complex)
  bool isAnyComplexType() const;   // C99 6.2.5p11 (complex) + Complex Int.
  bool isFloatingType() const;     // C99 6.2.5p11 (real floating + complex)
  bool isRealType() const;         // C99 6.2.5p17 (real floating + integer)
  bool isArithmeticType() const;   // C99 6.2.5p18 (integer + floating)
  bool isVoidType() const;         // C99 6.2.5p19
  bool isDerivedType() const;      // C99 6.2.5p20
  bool isScalarType() const;       // C99 6.2.5p21 (arithmetic + pointers)
  bool isAggregateType() const;    // C99 6.2.5p21 (arrays, structures)
  
  // Type Predicates: Check to see if this type is structurally the specified
  // type, ignoring typedefs and qualifiers.
  bool isFunctionType() const;
  bool isPointerLikeType() const; // Pointer or Reference.
  bool isPointerType() const;
  bool isReferenceType() const;
  bool isFunctionPointerType() const;
  bool isArrayType() const;
  bool isRecordType() const;
  bool isStructureType() const;   
  bool isUnionType() const;
  bool isComplexIntegerType() const;            // GCC _Complex integer type.
  bool isVectorType() const;                    // GCC vector type.
  bool isOCUVectorType() const;                 // OCU vector type.
  bool isObjCInterfaceType() const;             // NSString or NSString<foo>
  bool isObjCQualifiedInterfaceType() const;    // NSString<foo>
  bool isObjCQualifiedIdType() const;           // id<foo>
  
  // Type Checking Functions: Check to see if this type is structurally the
  // specified type, ignoring typedefs and qualifiers, and return a pointer to
  // the best type we can.
  const BuiltinType *getAsBuiltinType() const;   
  const FunctionType *getAsFunctionType() const;   
  const PointerLikeType *getAsPointerLikeType() const; // Pointer or Reference.
  const PointerType *getAsPointerType() const;
  const ReferenceType *getAsReferenceType() const;
  const ArrayType *getAsArrayType() const;
  const ConstantArrayType *getAsConstantArrayType() const;
  const VariableArrayType *getAsVariableArrayType() const;
  const IncompleteArrayType *getAsIncompleteArrayType() const;
  const RecordType *getAsRecordType() const;
  const RecordType *getAsStructureType() const;   
  const RecordType *getAsUnionType() const;
  const VectorType *getAsVectorType() const; // GCC vector type.
  const ComplexType *getAsComplexType() const;
  const ComplexType *getAsComplexIntegerType() const; // GCC complex int type.
  const OCUVectorType *getAsOCUVectorType() const; // OCU vector type.
  const ObjCInterfaceType *getAsObjCInterfaceType() const;
  const ObjCQualifiedInterfaceType *getAsObjCQualifiedInterfaceType() const;
  const ObjCQualifiedIdType *getAsObjCQualifiedIdType() const;

  
  /// getDesugaredType - Return the specified type with any "sugar" removed from
  /// type type.  This takes off typedefs, typeof's etc.  If the outer level of
  /// the type is already concrete, it returns it unmodified.  This is similar
  /// to getting the canonical type, but it doesn't remove *all* typedefs.  For
  /// example, it returns "T*" as "T*", (not as "int*"), because the pointer is
  /// concrete.
  const Type *getDesugaredType() const;
  
  /// More type predicates useful for type checking/promotion
  bool isPromotableIntegerType() const; // C99 6.3.1.1p2

  /// isSignedIntegerType - Return true if this is an integer type that is
  /// signed, according to C99 6.2.5p4 [char, signed char, short, int, long..],
  /// an enum decl which has a signed representation, or a vector of signed
  /// integer element type.
  bool isSignedIntegerType() const;

  /// isUnsignedIntegerType - Return true if this is an integer type that is
  /// unsigned, according to C99 6.2.5p6 [which returns true for _Bool], an enum
  /// decl which has an unsigned representation, or a vector of unsigned integer
  /// element type.
  bool isUnsignedIntegerType() const;

  /// isConstantSizeType - Return true if this is not a variable sized type,
  /// according to the rules of C99 6.7.5p3.  It is not legal to call this on
  /// incomplete types.
  bool isConstantSizeType() const;
private:  
  QualType getCanonicalTypeInternal() const { return CanonicalType; }
  friend class QualType;
public:
  virtual void getAsStringInternal(std::string &InnerString) const = 0;
  static bool classof(const Type *) { return true; }
  
protected:  
  /// Emit - Emit a Type to bitcode.  Used by ASTContext.
  void Emit(llvm::Serializer& S) const;
  
  /// Create - Construct a Type from bitcode.  Used by ASTContext.
  static void Create(ASTContext& Context, unsigned i, llvm::Deserializer& S);
  
  /// EmitImpl - Subclasses must implement this method in order to
  ///  be serialized.
  virtual void EmitImpl(llvm::Serializer& S) const;  
};

/// ASQualType - TR18037 (C embedded extensions) 6.2.5p26 
/// This supports address space qualified types.
///
class ASQualType : public Type, public llvm::FoldingSetNode {
  /// BaseType - This is the underlying type that this qualifies.  All CVR
  /// qualifiers are stored on the QualType that references this type, so we
  /// can't have any here.
  Type *BaseType;
  /// Address Space ID - The address space ID this type is qualified with.
  unsigned AddressSpace;
  ASQualType(Type *Base, QualType CanonicalPtr, unsigned AddrSpace) :
    Type(ASQual, CanonicalPtr), BaseType(Base), AddressSpace(AddrSpace) {
  }
  friend class ASTContext;  // ASTContext creates these.
public:
  Type *getBaseType() const { return BaseType; }
  unsigned getAddressSpace() const { return AddressSpace; }
  
  virtual void getAsStringInternal(std::string &InnerString) const;
  
  void Profile(llvm::FoldingSetNodeID &ID) {
    Profile(ID, getBaseType(), AddressSpace);
  }
  static void Profile(llvm::FoldingSetNodeID &ID, Type *Base, 
                      unsigned AddrSpace) {
    ID.AddPointer(Base);
    ID.AddInteger(AddrSpace);
  }
  
  static bool classof(const Type *T) { return T->getTypeClass() == ASQual; }
  static bool classof(const ASQualType *) { return true; }
  
protected:
  virtual void EmitImpl(llvm::Serializer& S) const;
  static Type* CreateImpl(ASTContext& Context,llvm::Deserializer& D);
  friend class Type;
};


/// BuiltinType - This class is used for builtin types like 'int'.  Builtin
/// types are always canonical and have a literal name field.
class BuiltinType : public Type {
public:
  enum Kind {
    Void,
    
    Bool,     // This is bool and/or _Bool.
    Char_U,   // This is 'char' for targets where char is unsigned.
    UChar,    // This is explicitly qualified unsigned char.
    UShort,
    UInt,
    ULong,
    ULongLong,
    
    Char_S,   // This is 'char' for targets where char is signed.
    SChar,    // This is explicitly qualified signed char.
    Short,
    Int,
    Long,
    LongLong,
    
    Float, Double, LongDouble
  };
private:
  Kind TypeKind;
public:
  BuiltinType(Kind K) : Type(Builtin, QualType()), TypeKind(K) {}
  
  Kind getKind() const { return TypeKind; }
  const char *getName() const;
  
  virtual void getAsStringInternal(std::string &InnerString) const;
  
  static bool classof(const Type *T) { return T->getTypeClass() == Builtin; }
  static bool classof(const BuiltinType *) { return true; }
};

/// ComplexType - C99 6.2.5p11 - Complex values.  This supports the C99 complex
/// types (_Complex float etc) as well as the GCC integer complex extensions.
///
class ComplexType : public Type, public llvm::FoldingSetNode {
  QualType ElementType;
  ComplexType(QualType Element, QualType CanonicalPtr) :
    Type(Complex, CanonicalPtr), ElementType(Element) {
  }
  friend class ASTContext;  // ASTContext creates these.
public:
  QualType getElementType() const { return ElementType; }
  
  virtual void getAsStringInternal(std::string &InnerString) const;
    
  void Profile(llvm::FoldingSetNodeID &ID) {
    Profile(ID, getElementType());
  }
  static void Profile(llvm::FoldingSetNodeID &ID, QualType Element) {
    ID.AddPointer(Element.getAsOpaquePtr());
  }
  
  static bool classof(const Type *T) { return T->getTypeClass() == Complex; }
  static bool classof(const ComplexType *) { return true; }
  
protected:  
  virtual void EmitImpl(llvm::Serializer& S) const;
  static Type* CreateImpl(ASTContext& Context,llvm::Deserializer& D);
  friend class Type;
};

/// PointerLikeType - Common base class for pointers and references.
///
class PointerLikeType : public Type {
  QualType PointeeType;
protected:
  PointerLikeType(TypeClass K, QualType Pointee, QualType CanonicalPtr) :
    Type(K, CanonicalPtr), PointeeType(Pointee) {
  }
public:
  
  QualType getPointeeType() const { return PointeeType; }

  static bool classof(const Type *T) {
    return T->getTypeClass() == Pointer || T->getTypeClass() == Reference;
  }
  static bool classof(const PointerLikeType *) { return true; }
};

/// PointerType - C99 6.7.5.1 - Pointer Declarators.
///
class PointerType : public PointerLikeType, public llvm::FoldingSetNode {
  PointerType(QualType Pointee, QualType CanonicalPtr) :
    PointerLikeType(Pointer, Pointee, CanonicalPtr) {
  }
  friend class ASTContext;  // ASTContext creates these.
public:
  
  virtual void getAsStringInternal(std::string &InnerString) const;
  
  void Profile(llvm::FoldingSetNodeID &ID) {
    Profile(ID, getPointeeType());
  }
  static void Profile(llvm::FoldingSetNodeID &ID, QualType Pointee) {
    ID.AddPointer(Pointee.getAsOpaquePtr());
  }
  
  static bool classof(const Type *T) { return T->getTypeClass() == Pointer; }
  static bool classof(const PointerType *) { return true; }

protected:  
  virtual void EmitImpl(llvm::Serializer& S) const;
  static Type* CreateImpl(ASTContext& Context,llvm::Deserializer& D);
  friend class Type;
};

/// ReferenceType - C++ 8.3.2 - Reference Declarators.
///
class ReferenceType : public PointerLikeType, public llvm::FoldingSetNode {
  ReferenceType(QualType Referencee, QualType CanonicalRef) :
    PointerLikeType(Reference, Referencee, CanonicalRef) {
  }
  friend class ASTContext;  // ASTContext creates these.
public:
  virtual void getAsStringInternal(std::string &InnerString) const;

  void Profile(llvm::FoldingSetNodeID &ID) {
    Profile(ID, getPointeeType());
  }
  static void Profile(llvm::FoldingSetNodeID &ID, QualType Referencee) {
    ID.AddPointer(Referencee.getAsOpaquePtr());
  }

  static bool classof(const Type *T) { return T->getTypeClass() == Reference; }
  static bool classof(const ReferenceType *) { return true; }
};

/// ArrayType - C99 6.7.5.2 - Array Declarators.
///
class ArrayType : public Type, public llvm::FoldingSetNode {
public:
  /// ArraySizeModifier - Capture whether this is a normal array (e.g. int X[4])
  /// an array with a static size (e.g. int X[static 4]), or with a star size
  /// (e.g. int X[*]). 'static' is only allowed on function parameters.
  enum ArraySizeModifier {
    Normal, Static, Star
  };
private:
  /// ElementType - The element type of the array.
  QualType ElementType;
  
  // NOTE: VC++ treats enums as signed, avoid using the ArraySizeModifier enum
  /// NOTE: These fields are packed into the bitfields space in the Type class.
  unsigned SizeModifier : 2;
  
  /// IndexTypeQuals - Capture qualifiers in declarations like:
  /// 'int X[static restrict 4]'. For function parameters only.
  unsigned IndexTypeQuals : 3;
  
protected:
  ArrayType(TypeClass tc, QualType et, QualType can,
            ArraySizeModifier sm, unsigned tq)
    : Type(tc, can), ElementType(et), SizeModifier(sm), IndexTypeQuals(tq) {}
  friend class ASTContext;  // ASTContext creates these.
public:
  QualType getElementType() const { return ElementType; }
  ArraySizeModifier getSizeModifier() const {
    return ArraySizeModifier(SizeModifier);
  }
  unsigned getIndexTypeQualifier() const { return IndexTypeQuals; }
  
  QualType getBaseType() const {
    const ArrayType *AT;
    QualType ElmtType = getElementType();
    // If we have a multi-dimensional array, navigate to the base type.
    while ((AT = ElmtType->getAsArrayType()))
      ElmtType = AT->getElementType();
    return ElmtType;
  }
  static bool classof(const Type *T) {
    return T->getTypeClass() == ConstantArray ||
           T->getTypeClass() == VariableArray ||
           T->getTypeClass() == IncompleteArray;
  }
  static bool classof(const ArrayType *) { return true; }
};

class ConstantArrayType : public ArrayType {
  llvm::APInt Size; // Allows us to unique the type.
  
  ConstantArrayType(QualType et, QualType can, llvm::APInt sz,
                    ArraySizeModifier sm, unsigned tq)
    : ArrayType(ConstantArray, et, can, sm, tq), Size(sz) {}
  friend class ASTContext;  // ASTContext creates these.
public:
  llvm::APInt getSize() const { return Size; }
  int getMaximumElements() const {
    QualType ElmtType = getElementType();
    int maxElements = static_cast<int>(getSize().getZExtValue());

    const ConstantArrayType *CAT;
    // If we have a multi-dimensional array, include it's elements.
    while ((CAT = ElmtType->getAsConstantArrayType())) {
      ElmtType = CAT->getElementType();
      maxElements *= static_cast<int>(CAT->getSize().getZExtValue());
    }
    return maxElements;
  }
  virtual void getAsStringInternal(std::string &InnerString) const;
  
  void Profile(llvm::FoldingSetNodeID &ID) {
    Profile(ID, getElementType(), getSize());
  }
  static void Profile(llvm::FoldingSetNodeID &ID, QualType ET,
                      llvm::APInt ArraySize) {
    ID.AddPointer(ET.getAsOpaquePtr());
    ID.AddInteger(ArraySize.getZExtValue());
  }
  static bool classof(const Type *T) { 
    return T->getTypeClass() == ConstantArray; 
  }
  static bool classof(const ConstantArrayType *) { return true; }
  
protected:  
  virtual void EmitImpl(llvm::Serializer& S) const;
  static Type* CreateImpl(ASTContext& Context, llvm::Deserializer& D);
  friend class Type;
};

class IncompleteArrayType : public ArrayType {
  IncompleteArrayType(QualType et, QualType can,
                    ArraySizeModifier sm, unsigned tq)
    : ArrayType(IncompleteArray, et, can, sm, tq) {}
  friend class ASTContext;  // ASTContext creates these.
public:

  virtual void getAsStringInternal(std::string &InnerString) const;

  static bool classof(const Type *T) { 
    return T->getTypeClass() == IncompleteArray; 
  }
  static bool classof(const IncompleteArrayType *) { return true; }
  
  friend class StmtIteratorBase;
  
  void Profile(llvm::FoldingSetNodeID &ID) {
    Profile(ID, getElementType());
  }
  
  static void Profile(llvm::FoldingSetNodeID &ID, QualType ET) {
    ID.AddPointer(ET.getAsOpaquePtr());
  }
  
protected:  
  virtual void EmitImpl(llvm::Serializer& S) const;
  static Type* CreateImpl(ASTContext& Context,llvm::Deserializer& D);
  friend class Type;
};

// FIXME: VariableArrayType's aren't uniqued (since expressions aren't).
class VariableArrayType : public ArrayType {
  /// SizeExpr - An assignment expression. VLA's are only permitted within 
  /// a function block. 
  Expr *SizeExpr;
  
  VariableArrayType(QualType et, QualType can, Expr *e,
                    ArraySizeModifier sm, unsigned tq)
    : ArrayType(VariableArray, et, can, sm, tq), SizeExpr(e) {}
  friend class ASTContext;  // ASTContext creates these.
public:
  const Expr *getSizeExpr() const { return SizeExpr; }
  Expr *getSizeExpr() { return SizeExpr; }
  
  virtual void getAsStringInternal(std::string &InnerString) const;
  
  static bool classof(const Type *T) { 
    return T->getTypeClass() == VariableArray; 
  }
  static bool classof(const VariableArrayType *) { return true; }
  
  friend class StmtIteratorBase;
  
  void Profile(llvm::FoldingSetNodeID &ID) {
    assert (0 && "Cannnot unique VariableArrayTypes.");
  }
  
protected:  
  virtual void EmitImpl(llvm::Serializer& S) const;
  static Type* CreateImpl(ASTContext& Context,llvm::Deserializer& D);
  friend class Type;
};

/// VectorType - GCC generic vector type. This type is created using
/// __attribute__((vector_size(n)), where "n" specifies the vector size in 
/// bytes. Since the constructor takes the number of vector elements, the 
/// client is responsible for converting the size into the number of elements.
class VectorType : public Type, public llvm::FoldingSetNode {
protected:
  /// ElementType - The element type of the vector.
  QualType ElementType;
  
  /// NumElements - The number of elements in the vector.
  unsigned NumElements;
  
  VectorType(QualType vecType, unsigned nElements, QualType canonType) :
    Type(Vector, canonType), ElementType(vecType), NumElements(nElements) {} 
  VectorType(TypeClass tc, QualType vecType, unsigned nElements, 
    QualType canonType) : Type(tc, canonType), ElementType(vecType), 
    NumElements(nElements) {} 
  friend class ASTContext;  // ASTContext creates these.
public:
    
  QualType getElementType() const { return ElementType; }
  unsigned getNumElements() const { return NumElements; } 

  virtual void getAsStringInternal(std::string &InnerString) const;
  
  void Profile(llvm::FoldingSetNodeID &ID) {
    Profile(ID, getElementType(), getNumElements(), getTypeClass());
  }
  static void Profile(llvm::FoldingSetNodeID &ID, QualType ElementType, 
                      unsigned NumElements, TypeClass TypeClass) {
    ID.AddPointer(ElementType.getAsOpaquePtr());
    ID.AddInteger(NumElements);
    ID.AddInteger(TypeClass);
  }
  static bool classof(const Type *T) { 
    return T->getTypeClass() == Vector || T->getTypeClass() == OCUVector; 
  }
  static bool classof(const VectorType *) { return true; }
};

/// OCUVectorType - Extended vector type. This type is created using
/// __attribute__((ocu_vector_type(n)), where "n" is the number of elements.
/// Unlike vector_size, ocu_vector_type is only allowed on typedef's. This
/// class enables syntactic extensions, like Vector Components for accessing
/// points, colors, and textures (modeled after OpenGL Shading Language).
class OCUVectorType : public VectorType {
  OCUVectorType(QualType vecType, unsigned nElements, QualType canonType) :
    VectorType(OCUVector, vecType, nElements, canonType) {} 
  friend class ASTContext;  // ASTContext creates these.
public:
  static int getPointAccessorIdx(char c) {
    switch (c) {
    default: return -1;
    case 'x': return 0;
    case 'y': return 1;
    case 'z': return 2;
    case 'w': return 3;
    }
  }
  static int getColorAccessorIdx(char c) {
    switch (c) {
    default: return -1;
    case 'r': return 0;
    case 'g': return 1;
    case 'b': return 2;
    case 'a': return 3;
    }
  }
  static int getTextureAccessorIdx(char c) {
    switch (c) {
    default: return -1;
    case 's': return 0;
    case 't': return 1;
    case 'p': return 2;
    case 'q': return 3;
    }
  };
  
  static int getAccessorIdx(char c) {
    if (int idx = getPointAccessorIdx(c)+1) return idx-1;
    if (int idx = getColorAccessorIdx(c)+1) return idx-1;
    return getTextureAccessorIdx(c);
  }
  
  bool isAccessorWithinNumElements(char c) const {
    if (int idx = getAccessorIdx(c)+1)
      return unsigned(idx-1) < NumElements;
    return false;
  }
  virtual void getAsStringInternal(std::string &InnerString) const;

  static bool classof(const Type *T) { 
    return T->getTypeClass() == OCUVector; 
  }
  static bool classof(const OCUVectorType *) { return true; }
};

/// FunctionType - C99 6.7.5.3 - Function Declarators.  This is the common base
/// class of FunctionTypeNoProto and FunctionTypeProto.
///
class FunctionType : public Type {
  /// SubClassData - This field is owned by the subclass, put here to pack
  /// tightly with the ivars in Type.
  bool SubClassData : 1;
  
  // The type returned by the function.
  QualType ResultType;
protected:
  FunctionType(TypeClass tc, QualType res, bool SubclassInfo,QualType Canonical)
    : Type(tc, Canonical), SubClassData(SubclassInfo), ResultType(res) {}
  bool getSubClassData() const { return SubClassData; }
public:
  
  QualType getResultType() const { return ResultType; }

  
  static bool classof(const Type *T) {
    return T->getTypeClass() == FunctionNoProto ||
           T->getTypeClass() == FunctionProto;
  }
  static bool classof(const FunctionType *) { return true; }
};

/// FunctionTypeNoProto - Represents a K&R-style 'int foo()' function, which has
/// no information available about its arguments.
class FunctionTypeNoProto : public FunctionType, public llvm::FoldingSetNode {
  FunctionTypeNoProto(QualType Result, QualType Canonical)
    : FunctionType(FunctionNoProto, Result, false, Canonical) {}
  friend class ASTContext;  // ASTContext creates these.
public:
  // No additional state past what FunctionType provides.
  
  virtual void getAsStringInternal(std::string &InnerString) const;

  void Profile(llvm::FoldingSetNodeID &ID) {
    Profile(ID, getResultType());
  }
  static void Profile(llvm::FoldingSetNodeID &ID, QualType ResultType) {
    ID.AddPointer(ResultType.getAsOpaquePtr());
  }
  
  static bool classof(const Type *T) {
    return T->getTypeClass() == FunctionNoProto;
  }
  static bool classof(const FunctionTypeNoProto *) { return true; }
  
protected:  
  virtual void EmitImpl(llvm::Serializer& S) const;
  static Type* CreateImpl(ASTContext& Context,llvm::Deserializer& D);
  friend class Type;
};

/// FunctionTypeProto - Represents a prototype with argument type info, e.g.
/// 'int foo(int)' or 'int foo(void)'.  'void' is represented as having no
/// arguments, not as having a single void argument.
class FunctionTypeProto : public FunctionType, public llvm::FoldingSetNode {
  FunctionTypeProto(QualType Result, QualType *ArgArray, unsigned numArgs,
                    bool isVariadic, QualType Canonical)
    : FunctionType(FunctionProto, Result, isVariadic, Canonical),
      NumArgs(numArgs) {
    // Fill in the trailing argument array.
    QualType *ArgInfo = reinterpret_cast<QualType *>(this+1);;
    for (unsigned i = 0; i != numArgs; ++i)
      ArgInfo[i] = ArgArray[i];
  }
  
  /// NumArgs - The number of arguments this function has, not counting '...'.
  unsigned NumArgs;
  
  /// ArgInfo - There is an variable size array after the class in memory that
  /// holds the argument types.
  friend class ASTContext;  // ASTContext creates these.
public:
  unsigned getNumArgs() const { return NumArgs; }
  QualType getArgType(unsigned i) const {
    assert(i < NumArgs && "Invalid argument number!");
    return arg_type_begin()[i];
  }
    
  bool isVariadic() const { return getSubClassData(); }
  
  typedef const QualType *arg_type_iterator;
  arg_type_iterator arg_type_begin() const {
    return reinterpret_cast<const QualType *>(this+1);
  }
  arg_type_iterator arg_type_end() const { return arg_type_begin()+NumArgs; }
  
  virtual void getAsStringInternal(std::string &InnerString) const;

  static bool classof(const Type *T) {
    return T->getTypeClass() == FunctionProto;
  }
  static bool classof(const FunctionTypeProto *) { return true; }
  
  void Profile(llvm::FoldingSetNodeID &ID);
  static void Profile(llvm::FoldingSetNodeID &ID, QualType Result,
                      arg_type_iterator ArgTys, unsigned NumArgs,
                      bool isVariadic);

protected:  
  virtual void EmitImpl(llvm::Serializer& S) const;
  static Type* CreateImpl(ASTContext& Context,llvm::Deserializer& D);
  friend class Type;
};


class TypedefType : public Type {
  TypedefDecl *Decl;
protected:
  TypedefType(TypeClass tc, TypedefDecl *D, QualType can) 
    : Type(tc, can), Decl(D) {
    assert(!isa<TypedefType>(can) && "Invalid canonical type");
  }
  friend class ASTContext;  // ASTContext creates these.
public:
  
  TypedefDecl *getDecl() const { return Decl; }
  
  /// LookThroughTypedefs - Return the ultimate type this typedef corresponds to
  /// potentially looking through *all* consequtive typedefs.  This returns the
  /// sum of the type qualifiers, so if you have:
  ///   typedef const int A;
  ///   typedef volatile A B;
  /// looking through the typedefs for B will give you "const volatile A".
  QualType LookThroughTypedefs() const;
    
  virtual void getAsStringInternal(std::string &InnerString) const;

  static bool classof(const Type *T) { return T->getTypeClass() == TypeName; }
  static bool classof(const TypedefType *) { return true; }
  
protected:  
  virtual void EmitImpl(llvm::Serializer& S) const;
  static Type* CreateImpl(ASTContext& Context,llvm::Deserializer& D);
  friend class Type;
};

/// TypeOfExpr (GCC extension).
class TypeOfExpr : public Type {
  Expr *TOExpr;
  TypeOfExpr(Expr *E, QualType can) : Type(TypeOfExp, can), TOExpr(E) {
    assert(!isa<TypedefType>(can) && "Invalid canonical type");
  }
  friend class ASTContext;  // ASTContext creates these.
public:
  Expr *getUnderlyingExpr() const { return TOExpr; }
  
  virtual void getAsStringInternal(std::string &InnerString) const;

  static bool classof(const Type *T) { return T->getTypeClass() == TypeOfExp; }
  static bool classof(const TypeOfExpr *) { return true; }
};

/// TypeOfType (GCC extension).
class TypeOfType : public Type {
  QualType TOType;
  TypeOfType(QualType T, QualType can) : Type(TypeOfTyp, can), TOType(T) {
    assert(!isa<TypedefType>(can) && "Invalid canonical type");
  }
  friend class ASTContext;  // ASTContext creates these.
public:
  QualType getUnderlyingType() const { return TOType; }
  
  virtual void getAsStringInternal(std::string &InnerString) const;

  static bool classof(const Type *T) { return T->getTypeClass() == TypeOfTyp; }
  static bool classof(const TypeOfType *) { return true; }
};

class TagType : public Type {
  TagDecl *decl;
  TagType(TagDecl *D, QualType can) : Type(Tagged, can), decl(D) {}
  friend class ASTContext;  // ASTContext creates these.
public:
    
  TagDecl *getDecl() const { return decl; }
  
  virtual void getAsStringInternal(std::string &InnerString) const;
  
  static bool classof(const Type *T) { return T->getTypeClass() == Tagged; }
  static bool classof(const TagType *) { return true; }
  
protected:  
  virtual void EmitImpl(llvm::Serializer& S) const;
  static Type* CreateImpl(ASTContext& Context, llvm::Deserializer& D);
  friend class Type;
};

/// RecordType - This is a helper class that allows the use of isa/cast/dyncast
/// to detect TagType objects of structs/unions/classes.
class RecordType : public TagType {
  RecordType(); // DO NOT IMPLEMENT
public:
    
  RecordDecl *getDecl() const {
    return reinterpret_cast<RecordDecl*>(TagType::getDecl());
  }
  
  // FIXME: This predicate is a helper to QualType/Type. It needs to 
  // recursively check all fields for const-ness. If any field is declared
  // const, it needs to return false. 
  bool hasConstFields() const { return false; }

  // FIXME: RecordType needs to check when it is created that all fields are in
  // the same address space, and return that.
  unsigned getAddressSpace() const { return 0; }
  
  static bool classof(const TagType *T);
  static bool classof(const Type *T) {
    return isa<TagType>(T) && classof(cast<TagType>(T));
  }
  static bool classof(const RecordType *) { return true; }
};

/// EnumType - This is a helper class that allows the use of isa/cast/dyncast
/// to detect TagType objects of enums.
class EnumType : public TagType {
  EnumType(); // DO NOT IMPLEMENT
public:
    
  EnumDecl *getDecl() const {
    return reinterpret_cast<EnumDecl*>(TagType::getDecl());
  }
  
  static bool classof(const TagType *T);
  static bool classof(const Type *T) {
    return isa<TagType>(T) && classof(cast<TagType>(T));
  }
  static bool classof(const EnumType *) { return true; }
};


  
  
class ObjCInterfaceType : public Type {
  ObjCInterfaceDecl *Decl;
protected:
  ObjCInterfaceType(TypeClass tc, ObjCInterfaceDecl *D) : 
    Type(tc, QualType()), Decl(D) { }
  friend class ASTContext;  // ASTContext creates these.
public:
  
  ObjCInterfaceDecl *getDecl() const { return Decl; }
  
  virtual void getAsStringInternal(std::string &InnerString) const;
  
  static bool classof(const Type *T) { 
    return T->getTypeClass() == ObjCInterface ||
           T->getTypeClass() == ObjCQualifiedInterface; 
  }
  static bool classof(const ObjCInterfaceType *) { return true; }
};

/// ObjCQualifiedInterfaceType - This class represents interface types 
/// conforming to a list of protocols, such as INTF<Proto1, Proto2, Proto1>.
///
/// Duplicate protocols are removed and protocol list is canonicalized to be in
/// alphabetical order.
class ObjCQualifiedInterfaceType : public ObjCInterfaceType, 
                                   public llvm::FoldingSetNode {
                                     
  // List of protocols for this protocol conforming object type
  // List is sorted on protocol name. No protocol is enterred more than once.
  llvm::SmallVector<ObjCProtocolDecl*, 8> Protocols;

  ObjCQualifiedInterfaceType(ObjCInterfaceDecl *D,
                             ObjCProtocolDecl **Protos, unsigned NumP) : 
    ObjCInterfaceType(ObjCQualifiedInterface, D), 
    Protocols(Protos, Protos+NumP) { }
  friend class ASTContext;  // ASTContext creates these.
public:
  
  ObjCProtocolDecl *getProtocols(unsigned i) const {
    return Protocols[i];
  }
  unsigned getNumProtocols() const {
    return Protocols.size();
  }

  typedef llvm::SmallVector<ObjCProtocolDecl*, 8>::const_iterator qual_iterator;
  qual_iterator qual_begin() const { return Protocols.begin(); }
  qual_iterator qual_end() const   { return Protocols.end(); }
                                     
  virtual void getAsStringInternal(std::string &InnerString) const;
  
  void Profile(llvm::FoldingSetNodeID &ID);
  static void Profile(llvm::FoldingSetNodeID &ID, 
                      const ObjCInterfaceDecl *Decl,
                      ObjCProtocolDecl **protocols, unsigned NumProtocols);
 
  static bool classof(const Type *T) { 
    return T->getTypeClass() == ObjCQualifiedInterface; 
  }
  static bool classof(const ObjCQualifiedInterfaceType *) { return true; }
};

/// ObjCQualifiedIdType - to represent id<protocol-list>.
///
/// Duplicate protocols are removed and protocol list is canonicalized to be in
/// alphabetical order.
class ObjCQualifiedIdType : public Type,
                            public llvm::FoldingSetNode {
  // List of protocols for this protocol conforming 'id' type
  // List is sorted on protocol name. No protocol is enterred more than once.
  llvm::SmallVector<ObjCProtocolDecl*, 8> Protocols;
    
  ObjCQualifiedIdType(QualType can, ObjCProtocolDecl **Protos,  unsigned NumP)
  : Type(ObjCQualifiedId, can), 
  Protocols(Protos, Protos+NumP) { }
  friend class ASTContext;  // ASTContext creates these.
public:
    
  ObjCProtocolDecl *getProtocols(unsigned i) const {
    return Protocols[i];
  }
  unsigned getNumProtocols() const {
    return Protocols.size();
  }
  ObjCProtocolDecl **getReferencedProtocols() {
    return &Protocols[0];
  }
                              
  typedef llvm::SmallVector<ObjCProtocolDecl*, 8>::const_iterator qual_iterator;
  qual_iterator qual_begin() const { return Protocols.begin(); }
  qual_iterator qual_end() const   { return Protocols.end(); }
    
  virtual void getAsStringInternal(std::string &InnerString) const;
    
  void Profile(llvm::FoldingSetNodeID &ID);
  static void Profile(llvm::FoldingSetNodeID &ID, 
                      ObjCProtocolDecl **protocols, unsigned NumProtocols);
    
  static bool classof(const Type *T) { 
    return T->getTypeClass() == ObjCQualifiedId; 
  }
  static bool classof(const ObjCQualifiedIdType *) { return true; }
    
};
  

// Inline function definitions.

/// getCanonicalType - Return the canonical version of this type, with the
/// appropriate type qualifiers on it.
inline QualType QualType::getCanonicalType() const {
  QualType CanType = getTypePtr()->getCanonicalTypeInternal();
  return QualType(CanType.getTypePtr(),
                  getCVRQualifiers() | CanType.getCVRQualifiers());
}

/// getUnqualifiedType - Return the type without any qualifiers.
inline QualType QualType::getUnqualifiedType() const {
  Type *TP = getTypePtr();
  if (const ASQualType *ASQT = dyn_cast<ASQualType>(TP))
    TP = ASQT->getBaseType();
  return QualType(TP, 0);
}

/// getAddressSpace - Return the address space of this type.
inline unsigned QualType::getAddressSpace() const {
  if (const ArrayType *AT = dyn_cast<ArrayType>(getCanonicalType()))
    return AT->getBaseType().getAddressSpace();
  if (const RecordType *RT = dyn_cast<RecordType>(getCanonicalType()))
    return RT->getAddressSpace();
  if (const ASQualType *ASQT = dyn_cast<ASQualType>(getCanonicalType()))
    return ASQT->getAddressSpace();
  return 0;
}

inline bool Type::isFunctionType() const {
  return isa<FunctionType>(CanonicalType.getUnqualifiedType());
}
inline bool Type::isPointerType() const {
  return isa<PointerType>(CanonicalType.getUnqualifiedType()); 
}
inline bool Type::isReferenceType() const {
  return isa<ReferenceType>(CanonicalType.getUnqualifiedType());
}
inline bool Type::isPointerLikeType() const {
  return isa<PointerLikeType>(CanonicalType.getUnqualifiedType()); 
}
inline bool Type::isFunctionPointerType() const {
  if (const PointerType* T = getAsPointerType())
    return T->getPointeeType()->isFunctionType();
  else
    return false;
}
inline bool Type::isArrayType() const {
  return isa<ArrayType>(CanonicalType.getUnqualifiedType());
}
inline bool Type::isRecordType() const {
  return isa<RecordType>(CanonicalType.getUnqualifiedType());
}
inline bool Type::isAnyComplexType() const {
  return isa<ComplexType>(CanonicalType);
}
inline bool Type::isVectorType() const {
  return isa<VectorType>(CanonicalType.getUnqualifiedType());
}
inline bool Type::isOCUVectorType() const {
  return isa<OCUVectorType>(CanonicalType.getUnqualifiedType());
}
inline bool Type::isObjCInterfaceType() const {
  return isa<ObjCInterfaceType>(CanonicalType);
}
inline bool Type::isObjCQualifiedInterfaceType() const {
  return isa<ObjCQualifiedInterfaceType>(CanonicalType);
}
inline bool Type::isObjCQualifiedIdType() const {
  return isa<ObjCQualifiedIdType>(CanonicalType);
}
}  // end namespace clang

#endif
