//===----------------------------------------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

// <string_view>

// constexpr int compare(size_type pos1, size_type n1, basic_string_view str,
//             size_type pos2, size_type n2) const;

#include <string_view>
#include <cassert>
#include <stdexcept>

#include "test_macros.h"
#include "constexpr_char_traits.hpp"

int sign ( int x ) { return x > 0 ? 1 : ( x < 0 ? -1 : 0 ); }

template<typename CharT>
void test1 ( std::basic_string_view<CharT> sv1, size_t pos1, size_t n1,
             std::basic_string_view<CharT> sv2, size_t pos2, size_t n2,
             int expected ) {
#ifdef TEST_HAS_NO_EXCEPTIONS
    if (pos1 <= sv1.size() && pos2 <= sv2.size())
        assert (sign( sv1.compare(pos1, n1, sv2, pos2, n2)) == sign(expected));
#else
    try {
        assert (sign( sv1.compare(pos1, n1, sv2, pos2, n2)) == sign(expected));
        assert(pos1 <= sv1.size() && pos2 <= sv2.size());
        }
    catch (const std::out_of_range&) {
        assert(pos1 > sv1.size() || pos2 > sv2.size());
        }
#endif
}


template<typename CharT>
void test ( const CharT *s1, size_t pos1, size_t n1,
            const CharT *s2, size_t pos2, size_t n2,
            int expected ) {
    typedef std::basic_string_view<CharT> string_view_t;

    string_view_t sv1 ( s1 );
    string_view_t sv2 ( s2 );
    test1(sv1, pos1, n1, sv2, pos2, n2, expected);
}

void test0()
{
    test("", 0, 0, "", 0, 0, 0);
    test("", 0, 0, "", 0, 1, 0);
    test("", 0, 0, "", 1, 0, 0);
    test("", 0, 0, "abcde", 0, 0, 0);
    test("", 0, 0, "abcde", 0, 1, -1);
    test("", 0, 0, "abcde", 0, 2, -2);
    test("", 0, 0, "abcde", 0, 4, -4);
    test("", 0, 0, "abcde", 0, 5, -5);
    test("", 0, 0, "abcde", 0, 6, -5);
    test("", 0, 0, "abcde", 1, 0, 0);
    test("", 0, 0, "abcde", 1, 1, -1);
    test("", 0, 0, "abcde", 1, 2, -2);
    test("", 0, 0, "abcde", 1, 3, -3);
    test("", 0, 0, "abcde", 1, 4, -4);
    test("", 0, 0, "abcde", 1, 5, -4);
    test("", 0, 0, "abcde", 2, 0, 0);
    test("", 0, 0, "abcde", 2, 1, -1);
    test("", 0, 0, "abcde", 2, 2, -2);
    test("", 0, 0, "abcde", 2, 3, -3);
    test("", 0, 0, "abcde", 2, 4, -3);
    test("", 0, 0, "abcde", 4, 0, 0);
    test("", 0, 0, "abcde", 4, 1, -1);
    test("", 0, 0, "abcde", 4, 2, -1);
    test("", 0, 0, "abcde", 5, 0, 0);
    test("", 0, 0, "abcde", 5, 1, 0);
    test("", 0, 0, "abcde", 6, 0, 0);
    test("", 0, 0, "abcdefghij", 0, 0, 0);
    test("", 0, 0, "abcdefghij", 0, 1, -1);
    test("", 0, 0, "abcdefghij", 0, 5, -5);
    test("", 0, 0, "abcdefghij", 0, 9, -9);
    test("", 0, 0, "abcdefghij", 0, 10, -10);
    test("", 0, 0, "abcdefghij", 0, 11, -10);
    test("", 0, 0, "abcdefghij", 1, 0, 0);
    test("", 0, 0, "abcdefghij", 1, 1, -1);
    test("", 0, 0, "abcdefghij", 1, 4, -4);
    test("", 0, 0, "abcdefghij", 1, 8, -8);
    test("", 0, 0, "abcdefghij", 1, 9, -9);
    test("", 0, 0, "abcdefghij", 1, 10, -9);
    test("", 0, 0, "abcdefghij", 5, 0, 0);
    test("", 0, 0, "abcdefghij", 5, 1, -1);
    test("", 0, 0, "abcdefghij", 5, 2, -2);
    test("", 0, 0, "abcdefghij", 5, 4, -4);
    test("", 0, 0, "abcdefghij", 5, 5, -5);
    test("", 0, 0, "abcdefghij", 5, 6, -5);
    test("", 0, 0, "abcdefghij", 9, 0, 0);
    test("", 0, 0, "abcdefghij", 9, 1, -1);
    test("", 0, 0, "abcdefghij", 9, 2, -1);
    test("", 0, 0, "abcdefghij", 10, 0, 0);
    test("", 0, 0, "abcdefghij", 10, 1, 0);
    test("", 0, 0, "abcdefghij", 11, 0, 0);
    test("", 0, 0, "abcdefghijklmnopqrst", 0, 0, 0);
    test("", 0, 0, "abcdefghijklmnopqrst", 0, 1, -1);
    test("", 0, 0, "abcdefghijklmnopqrst", 0, 10, -10);
    test("", 0, 0, "abcdefghijklmnopqrst", 0, 19, -19);
    test("", 0, 0, "abcdefghijklmnopqrst", 0, 20, -20);
    test("", 0, 0, "abcdefghijklmnopqrst", 0, 21, -20);
    test("", 0, 0, "abcdefghijklmnopqrst", 1, 0, 0);
    test("", 0, 0, "abcdefghijklmnopqrst", 1, 1, -1);
    test("", 0, 0, "abcdefghijklmnopqrst", 1, 9, -9);
    test("", 0, 0, "abcdefghijklmnopqrst", 1, 18, -18);
    test("", 0, 0, "abcdefghijklmnopqrst", 1, 19, -19);
    test("", 0, 0, "abcdefghijklmnopqrst", 1, 20, -19);
    test("", 0, 0, "abcdefghijklmnopqrst", 10, 0, 0);
    test("", 0, 0, "abcdefghijklmnopqrst", 10, 1, -1);
    test("", 0, 0, "abcdefghijklmnopqrst", 10, 5, -5);
    test("", 0, 0, "abcdefghijklmnopqrst", 10, 9, -9);
    test("", 0, 0, "abcdefghijklmnopqrst", 10, 10, -10);
    test("", 0, 0, "abcdefghijklmnopqrst", 10, 11, -10);
    test("", 0, 0, "abcdefghijklmnopqrst", 19, 0, 0);
    test("", 0, 0, "abcdefghijklmnopqrst", 19, 1, -1);
    test("", 0, 0, "abcdefghijklmnopqrst", 19, 2, -1);
    test("", 0, 0, "abcdefghijklmnopqrst", 20, 0, 0);
    test("", 0, 0, "abcdefghijklmnopqrst", 20, 1, 0);
    test("", 0, 0, "abcdefghijklmnopqrst", 21, 0, 0);
    test("", 0, 1, "", 0, 0, 0);
    test("", 0, 1, "", 0, 1, 0);
    test("", 0, 1, "", 1, 0, 0);
    test("", 0, 1, "abcde", 0, 0, 0);
    test("", 0, 1, "abcde", 0, 1, -1);
    test("", 0, 1, "abcde", 0, 2, -2);
    test("", 0, 1, "abcde", 0, 4, -4);
    test("", 0, 1, "abcde", 0, 5, -5);
    test("", 0, 1, "abcde", 0, 6, -5);
    test("", 0, 1, "abcde", 1, 0, 0);
    test("", 0, 1, "abcde", 1, 1, -1);
    test("", 0, 1, "abcde", 1, 2, -2);
    test("", 0, 1, "abcde", 1, 3, -3);
    test("", 0, 1, "abcde", 1, 4, -4);
    test("", 0, 1, "abcde", 1, 5, -4);
    test("", 0, 1, "abcde", 2, 0, 0);
    test("", 0, 1, "abcde", 2, 1, -1);
    test("", 0, 1, "abcde", 2, 2, -2);
    test("", 0, 1, "abcde", 2, 3, -3);
    test("", 0, 1, "abcde", 2, 4, -3);
    test("", 0, 1, "abcde", 4, 0, 0);
    test("", 0, 1, "abcde", 4, 1, -1);
    test("", 0, 1, "abcde", 4, 2, -1);
    test("", 0, 1, "abcde", 5, 0, 0);
    test("", 0, 1, "abcde", 5, 1, 0);
    test("", 0, 1, "abcde", 6, 0, 0);
}

void test1()
{
    test("", 0, 1, "abcdefghij", 0, 0, 0);
    test("", 0, 1, "abcdefghij", 0, 1, -1);
    test("", 0, 1, "abcdefghij", 0, 5, -5);
    test("", 0, 1, "abcdefghij", 0, 9, -9);
    test("", 0, 1, "abcdefghij", 0, 10, -10);
    test("", 0, 1, "abcdefghij", 0, 11, -10);
    test("", 0, 1, "abcdefghij", 1, 0, 0);
    test("", 0, 1, "abcdefghij", 1, 1, -1);
    test("", 0, 1, "abcdefghij", 1, 4, -4);
    test("", 0, 1, "abcdefghij", 1, 8, -8);
    test("", 0, 1, "abcdefghij", 1, 9, -9);
    test("", 0, 1, "abcdefghij", 1, 10, -9);
    test("", 0, 1, "abcdefghij", 5, 0, 0);
    test("", 0, 1, "abcdefghij", 5, 1, -1);
    test("", 0, 1, "abcdefghij", 5, 2, -2);
    test("", 0, 1, "abcdefghij", 5, 4, -4);
    test("", 0, 1, "abcdefghij", 5, 5, -5);
    test("", 0, 1, "abcdefghij", 5, 6, -5);
    test("", 0, 1, "abcdefghij", 9, 0, 0);
    test("", 0, 1, "abcdefghij", 9, 1, -1);
    test("", 0, 1, "abcdefghij", 9, 2, -1);
    test("", 0, 1, "abcdefghij", 10, 0, 0);
    test("", 0, 1, "abcdefghij", 10, 1, 0);
    test("", 0, 1, "abcdefghij", 11, 0, 0);
    test("", 0, 1, "abcdefghijklmnopqrst", 0, 0, 0);
    test("", 0, 1, "abcdefghijklmnopqrst", 0, 1, -1);
    test("", 0, 1, "abcdefghijklmnopqrst", 0, 10, -10);
    test("", 0, 1, "abcdefghijklmnopqrst", 0, 19, -19);
    test("", 0, 1, "abcdefghijklmnopqrst", 0, 20, -20);
    test("", 0, 1, "abcdefghijklmnopqrst", 0, 21, -20);
    test("", 0, 1, "abcdefghijklmnopqrst", 1, 0, 0);
    test("", 0, 1, "abcdefghijklmnopqrst", 1, 1, -1);
    test("", 0, 1, "abcdefghijklmnopqrst", 1, 9, -9);
    test("", 0, 1, "abcdefghijklmnopqrst", 1, 18, -18);
    test("", 0, 1, "abcdefghijklmnopqrst", 1, 19, -19);
    test("", 0, 1, "abcdefghijklmnopqrst", 1, 20, -19);
    test("", 0, 1, "abcdefghijklmnopqrst", 10, 0, 0);
    test("", 0, 1, "abcdefghijklmnopqrst", 10, 1, -1);
    test("", 0, 1, "abcdefghijklmnopqrst", 10, 5, -5);
    test("", 0, 1, "abcdefghijklmnopqrst", 10, 9, -9);
    test("", 0, 1, "abcdefghijklmnopqrst", 10, 10, -10);
    test("", 0, 1, "abcdefghijklmnopqrst", 10, 11, -10);
    test("", 0, 1, "abcdefghijklmnopqrst", 19, 0, 0);
    test("", 0, 1, "abcdefghijklmnopqrst", 19, 1, -1);
    test("", 0, 1, "abcdefghijklmnopqrst", 19, 2, -1);
    test("", 0, 1, "abcdefghijklmnopqrst", 20, 0, 0);
    test("", 0, 1, "abcdefghijklmnopqrst", 20, 1, 0);
    test("", 0, 1, "abcdefghijklmnopqrst", 21, 0, 0);
    test("", 1, 0, "", 0, 0, 0);
    test("", 1, 0, "", 0, 1, 0);
    test("", 1, 0, "", 1, 0, 0);
    test("", 1, 0, "abcde", 0, 0, 0);
    test("", 1, 0, "abcde", 0, 1, 0);
    test("", 1, 0, "abcde", 0, 2, 0);
    test("", 1, 0, "abcde", 0, 4, 0);
    test("", 1, 0, "abcde", 0, 5, 0);
    test("", 1, 0, "abcde", 0, 6, 0);
    test("", 1, 0, "abcde", 1, 0, 0);
    test("", 1, 0, "abcde", 1, 1, 0);
    test("", 1, 0, "abcde", 1, 2, 0);
    test("", 1, 0, "abcde", 1, 3, 0);
    test("", 1, 0, "abcde", 1, 4, 0);
    test("", 1, 0, "abcde", 1, 5, 0);
    test("", 1, 0, "abcde", 2, 0, 0);
    test("", 1, 0, "abcde", 2, 1, 0);
    test("", 1, 0, "abcde", 2, 2, 0);
    test("", 1, 0, "abcde", 2, 3, 0);
    test("", 1, 0, "abcde", 2, 4, 0);
    test("", 1, 0, "abcde", 4, 0, 0);
    test("", 1, 0, "abcde", 4, 1, 0);
    test("", 1, 0, "abcde", 4, 2, 0);
    test("", 1, 0, "abcde", 5, 0, 0);
    test("", 1, 0, "abcde", 5, 1, 0);
    test("", 1, 0, "abcde", 6, 0, 0);
    test("", 1, 0, "abcdefghij", 0, 0, 0);
    test("", 1, 0, "abcdefghij", 0, 1, 0);
    test("", 1, 0, "abcdefghij", 0, 5, 0);
    test("", 1, 0, "abcdefghij", 0, 9, 0);
    test("", 1, 0, "abcdefghij", 0, 10, 0);
    test("", 1, 0, "abcdefghij", 0, 11, 0);
    test("", 1, 0, "abcdefghij", 1, 0, 0);
    test("", 1, 0, "abcdefghij", 1, 1, 0);
    test("", 1, 0, "abcdefghij", 1, 4, 0);
    test("", 1, 0, "abcdefghij", 1, 8, 0);
    test("", 1, 0, "abcdefghij", 1, 9, 0);
    test("", 1, 0, "abcdefghij", 1, 10, 0);
    test("", 1, 0, "abcdefghij", 5, 0, 0);
    test("", 1, 0, "abcdefghij", 5, 1, 0);
    test("", 1, 0, "abcdefghij", 5, 2, 0);
    test("", 1, 0, "abcdefghij", 5, 4, 0);
    test("", 1, 0, "abcdefghij", 5, 5, 0);
    test("", 1, 0, "abcdefghij", 5, 6, 0);
    test("", 1, 0, "abcdefghij", 9, 0, 0);
    test("", 1, 0, "abcdefghij", 9, 1, 0);
    test("", 1, 0, "abcdefghij", 9, 2, 0);
    test("", 1, 0, "abcdefghij", 10, 0, 0);
    test("", 1, 0, "abcdefghij", 10, 1, 0);
    test("", 1, 0, "abcdefghij", 11, 0, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 0, 0, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 0, 1, 0);
}

void test2()
{
    test("", 1, 0, "abcdefghijklmnopqrst", 0, 10, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 0, 19, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 0, 20, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 0, 21, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 1, 0, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 1, 1, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 1, 9, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 1, 18, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 1, 19, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 1, 20, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 10, 0, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 10, 1, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 10, 5, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 10, 9, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 10, 10, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 10, 11, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 19, 0, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 19, 1, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 19, 2, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 20, 0, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 20, 1, 0);
    test("", 1, 0, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 0, 0, "", 0, 0, 0);
    test("abcde", 0, 0, "", 0, 1, 0);
    test("abcde", 0, 0, "", 1, 0, 0);
    test("abcde", 0, 0, "abcde", 0, 0, 0);
    test("abcde", 0, 0, "abcde", 0, 1, -1);
    test("abcde", 0, 0, "abcde", 0, 2, -2);
    test("abcde", 0, 0, "abcde", 0, 4, -4);
    test("abcde", 0, 0, "abcde", 0, 5, -5);
    test("abcde", 0, 0, "abcde", 0, 6, -5);
    test("abcde", 0, 0, "abcde", 1, 0, 0);
    test("abcde", 0, 0, "abcde", 1, 1, -1);
    test("abcde", 0, 0, "abcde", 1, 2, -2);
    test("abcde", 0, 0, "abcde", 1, 3, -3);
    test("abcde", 0, 0, "abcde", 1, 4, -4);
    test("abcde", 0, 0, "abcde", 1, 5, -4);
    test("abcde", 0, 0, "abcde", 2, 0, 0);
    test("abcde", 0, 0, "abcde", 2, 1, -1);
    test("abcde", 0, 0, "abcde", 2, 2, -2);
    test("abcde", 0, 0, "abcde", 2, 3, -3);
    test("abcde", 0, 0, "abcde", 2, 4, -3);
    test("abcde", 0, 0, "abcde", 4, 0, 0);
    test("abcde", 0, 0, "abcde", 4, 1, -1);
    test("abcde", 0, 0, "abcde", 4, 2, -1);
    test("abcde", 0, 0, "abcde", 5, 0, 0);
    test("abcde", 0, 0, "abcde", 5, 1, 0);
    test("abcde", 0, 0, "abcde", 6, 0, 0);
    test("abcde", 0, 0, "abcdefghij", 0, 0, 0);
    test("abcde", 0, 0, "abcdefghij", 0, 1, -1);
    test("abcde", 0, 0, "abcdefghij", 0, 5, -5);
    test("abcde", 0, 0, "abcdefghij", 0, 9, -9);
    test("abcde", 0, 0, "abcdefghij", 0, 10, -10);
    test("abcde", 0, 0, "abcdefghij", 0, 11, -10);
    test("abcde", 0, 0, "abcdefghij", 1, 0, 0);
    test("abcde", 0, 0, "abcdefghij", 1, 1, -1);
    test("abcde", 0, 0, "abcdefghij", 1, 4, -4);
    test("abcde", 0, 0, "abcdefghij", 1, 8, -8);
    test("abcde", 0, 0, "abcdefghij", 1, 9, -9);
    test("abcde", 0, 0, "abcdefghij", 1, 10, -9);
    test("abcde", 0, 0, "abcdefghij", 5, 0, 0);
    test("abcde", 0, 0, "abcdefghij", 5, 1, -1);
    test("abcde", 0, 0, "abcdefghij", 5, 2, -2);
    test("abcde", 0, 0, "abcdefghij", 5, 4, -4);
    test("abcde", 0, 0, "abcdefghij", 5, 5, -5);
    test("abcde", 0, 0, "abcdefghij", 5, 6, -5);
    test("abcde", 0, 0, "abcdefghij", 9, 0, 0);
    test("abcde", 0, 0, "abcdefghij", 9, 1, -1);
    test("abcde", 0, 0, "abcdefghij", 9, 2, -1);
    test("abcde", 0, 0, "abcdefghij", 10, 0, 0);
    test("abcde", 0, 0, "abcdefghij", 10, 1, 0);
    test("abcde", 0, 0, "abcdefghij", 11, 0, 0);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 0, 0, 0);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 0, 1, -1);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 0, 10, -10);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 0, 19, -19);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 0, 20, -20);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 0, 21, -20);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 1, 0, 0);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 1, 9, -9);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 1, 18, -18);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 1, 19, -19);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 1, 20, -19);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 10, 0, 0);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 10, 1, -1);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 19, 0, 0);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 19, 1, -1);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 19, 2, -1);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 20, 0, 0);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 20, 1, 0);
    test("abcde", 0, 0, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 0, 1, "", 0, 0, 1);
    test("abcde", 0, 1, "", 0, 1, 1);
    test("abcde", 0, 1, "", 1, 0, 0);
    test("abcde", 0, 1, "abcde", 0, 0, 1);
}

void test3()
{
    test("abcde", 0, 1, "abcde", 0, 1, 0);
    test("abcde", 0, 1, "abcde", 0, 2, -1);
    test("abcde", 0, 1, "abcde", 0, 4, -3);
    test("abcde", 0, 1, "abcde", 0, 5, -4);
    test("abcde", 0, 1, "abcde", 0, 6, -4);
    test("abcde", 0, 1, "abcde", 1, 0, 1);
    test("abcde", 0, 1, "abcde", 1, 1, -1);
    test("abcde", 0, 1, "abcde", 1, 2, -1);
    test("abcde", 0, 1, "abcde", 1, 3, -1);
    test("abcde", 0, 1, "abcde", 1, 4, -1);
    test("abcde", 0, 1, "abcde", 1, 5, -1);
    test("abcde", 0, 1, "abcde", 2, 0, 1);
    test("abcde", 0, 1, "abcde", 2, 1, -2);
    test("abcde", 0, 1, "abcde", 2, 2, -2);
    test("abcde", 0, 1, "abcde", 2, 3, -2);
    test("abcde", 0, 1, "abcde", 2, 4, -2);
    test("abcde", 0, 1, "abcde", 4, 0, 1);
    test("abcde", 0, 1, "abcde", 4, 1, -4);
    test("abcde", 0, 1, "abcde", 4, 2, -4);
    test("abcde", 0, 1, "abcde", 5, 0, 1);
    test("abcde", 0, 1, "abcde", 5, 1, 1);
    test("abcde", 0, 1, "abcde", 6, 0, 0);
    test("abcde", 0, 1, "abcdefghij", 0, 0, 1);
    test("abcde", 0, 1, "abcdefghij", 0, 1, 0);
    test("abcde", 0, 1, "abcdefghij", 0, 5, -4);
    test("abcde", 0, 1, "abcdefghij", 0, 9, -8);
    test("abcde", 0, 1, "abcdefghij", 0, 10, -9);
    test("abcde", 0, 1, "abcdefghij", 0, 11, -9);
    test("abcde", 0, 1, "abcdefghij", 1, 0, 1);
    test("abcde", 0, 1, "abcdefghij", 1, 1, -1);
    test("abcde", 0, 1, "abcdefghij", 1, 4, -1);
    test("abcde", 0, 1, "abcdefghij", 1, 8, -1);
    test("abcde", 0, 1, "abcdefghij", 1, 9, -1);
    test("abcde", 0, 1, "abcdefghij", 1, 10, -1);
    test("abcde", 0, 1, "abcdefghij", 5, 0, 1);
    test("abcde", 0, 1, "abcdefghij", 5, 1, -5);
    test("abcde", 0, 1, "abcdefghij", 5, 2, -5);
    test("abcde", 0, 1, "abcdefghij", 5, 4, -5);
    test("abcde", 0, 1, "abcdefghij", 5, 5, -5);
    test("abcde", 0, 1, "abcdefghij", 5, 6, -5);
    test("abcde", 0, 1, "abcdefghij", 9, 0, 1);
    test("abcde", 0, 1, "abcdefghij", 9, 1, -9);
    test("abcde", 0, 1, "abcdefghij", 9, 2, -9);
    test("abcde", 0, 1, "abcdefghij", 10, 0, 1);
    test("abcde", 0, 1, "abcdefghij", 10, 1, 1);
    test("abcde", 0, 1, "abcdefghij", 11, 0, 0);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 0, 0, 1);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 0, 1, 0);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 0, 10, -9);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 0, 19, -18);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 0, 20, -19);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 0, 21, -19);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 1, 0, 1);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 1, 9, -1);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 1, 18, -1);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 1, 19, -1);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 1, 20, -1);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 10, 0, 1);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 10, 1, -10);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 10, 5, -10);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 10, 9, -10);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 19, 0, 1);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 19, 1, -19);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 19, 2, -19);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 20, 0, 1);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 20, 1, 1);
    test("abcde", 0, 1, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 0, 2, "", 0, 0, 2);
    test("abcde", 0, 2, "", 0, 1, 2);
    test("abcde", 0, 2, "", 1, 0, 0);
    test("abcde", 0, 2, "abcde", 0, 0, 2);
    test("abcde", 0, 2, "abcde", 0, 1, 1);
    test("abcde", 0, 2, "abcde", 0, 2, 0);
    test("abcde", 0, 2, "abcde", 0, 4, -2);
    test("abcde", 0, 2, "abcde", 0, 5, -3);
    test("abcde", 0, 2, "abcde", 0, 6, -3);
    test("abcde", 0, 2, "abcde", 1, 0, 2);
    test("abcde", 0, 2, "abcde", 1, 1, -1);
    test("abcde", 0, 2, "abcde", 1, 2, -1);
    test("abcde", 0, 2, "abcde", 1, 3, -1);
    test("abcde", 0, 2, "abcde", 1, 4, -1);
    test("abcde", 0, 2, "abcde", 1, 5, -1);
    test("abcde", 0, 2, "abcde", 2, 0, 2);
    test("abcde", 0, 2, "abcde", 2, 1, -2);
    test("abcde", 0, 2, "abcde", 2, 2, -2);
    test("abcde", 0, 2, "abcde", 2, 3, -2);
    test("abcde", 0, 2, "abcde", 2, 4, -2);
    test("abcde", 0, 2, "abcde", 4, 0, 2);
    test("abcde", 0, 2, "abcde", 4, 1, -4);
    test("abcde", 0, 2, "abcde", 4, 2, -4);
    test("abcde", 0, 2, "abcde", 5, 0, 2);
    test("abcde", 0, 2, "abcde", 5, 1, 2);
    test("abcde", 0, 2, "abcde", 6, 0, 0);
    test("abcde", 0, 2, "abcdefghij", 0, 0, 2);
    test("abcde", 0, 2, "abcdefghij", 0, 1, 1);
    test("abcde", 0, 2, "abcdefghij", 0, 5, -3);
    test("abcde", 0, 2, "abcdefghij", 0, 9, -7);
}

void test4()
{
    test("abcde", 0, 2, "abcdefghij", 0, 10, -8);
    test("abcde", 0, 2, "abcdefghij", 0, 11, -8);
    test("abcde", 0, 2, "abcdefghij", 1, 0, 2);
    test("abcde", 0, 2, "abcdefghij", 1, 1, -1);
    test("abcde", 0, 2, "abcdefghij", 1, 4, -1);
    test("abcde", 0, 2, "abcdefghij", 1, 8, -1);
    test("abcde", 0, 2, "abcdefghij", 1, 9, -1);
    test("abcde", 0, 2, "abcdefghij", 1, 10, -1);
    test("abcde", 0, 2, "abcdefghij", 5, 0, 2);
    test("abcde", 0, 2, "abcdefghij", 5, 1, -5);
    test("abcde", 0, 2, "abcdefghij", 5, 2, -5);
    test("abcde", 0, 2, "abcdefghij", 5, 4, -5);
    test("abcde", 0, 2, "abcdefghij", 5, 5, -5);
    test("abcde", 0, 2, "abcdefghij", 5, 6, -5);
    test("abcde", 0, 2, "abcdefghij", 9, 0, 2);
    test("abcde", 0, 2, "abcdefghij", 9, 1, -9);
    test("abcde", 0, 2, "abcdefghij", 9, 2, -9);
    test("abcde", 0, 2, "abcdefghij", 10, 0, 2);
    test("abcde", 0, 2, "abcdefghij", 10, 1, 2);
    test("abcde", 0, 2, "abcdefghij", 11, 0, 0);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 0, 0, 2);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 0, 1, 1);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 0, 10, -8);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 0, 19, -17);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 0, 20, -18);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 0, 21, -18);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 1, 0, 2);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 1, 9, -1);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 1, 18, -1);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 1, 19, -1);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 1, 20, -1);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 10, 0, 2);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 10, 1, -10);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 10, 5, -10);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 10, 9, -10);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 19, 0, 2);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 19, 1, -19);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 19, 2, -19);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 20, 0, 2);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 20, 1, 2);
    test("abcde", 0, 2, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 0, 4, "", 0, 0, 4);
    test("abcde", 0, 4, "", 0, 1, 4);
    test("abcde", 0, 4, "", 1, 0, 0);
    test("abcde", 0, 4, "abcde", 0, 0, 4);
    test("abcde", 0, 4, "abcde", 0, 1, 3);
    test("abcde", 0, 4, "abcde", 0, 2, 2);
    test("abcde", 0, 4, "abcde", 0, 4, 0);
    test("abcde", 0, 4, "abcde", 0, 5, -1);
    test("abcde", 0, 4, "abcde", 0, 6, -1);
    test("abcde", 0, 4, "abcde", 1, 0, 4);
    test("abcde", 0, 4, "abcde", 1, 1, -1);
    test("abcde", 0, 4, "abcde", 1, 2, -1);
    test("abcde", 0, 4, "abcde", 1, 3, -1);
    test("abcde", 0, 4, "abcde", 1, 4, -1);
    test("abcde", 0, 4, "abcde", 1, 5, -1);
    test("abcde", 0, 4, "abcde", 2, 0, 4);
    test("abcde", 0, 4, "abcde", 2, 1, -2);
    test("abcde", 0, 4, "abcde", 2, 2, -2);
    test("abcde", 0, 4, "abcde", 2, 3, -2);
    test("abcde", 0, 4, "abcde", 2, 4, -2);
    test("abcde", 0, 4, "abcde", 4, 0, 4);
    test("abcde", 0, 4, "abcde", 4, 1, -4);
    test("abcde", 0, 4, "abcde", 4, 2, -4);
    test("abcde", 0, 4, "abcde", 5, 0, 4);
    test("abcde", 0, 4, "abcde", 5, 1, 4);
    test("abcde", 0, 4, "abcde", 6, 0, 0);
    test("abcde", 0, 4, "abcdefghij", 0, 0, 4);
    test("abcde", 0, 4, "abcdefghij", 0, 1, 3);
    test("abcde", 0, 4, "abcdefghij", 0, 5, -1);
    test("abcde", 0, 4, "abcdefghij", 0, 9, -5);
    test("abcde", 0, 4, "abcdefghij", 0, 10, -6);
    test("abcde", 0, 4, "abcdefghij", 0, 11, -6);
    test("abcde", 0, 4, "abcdefghij", 1, 0, 4);
    test("abcde", 0, 4, "abcdefghij", 1, 1, -1);
    test("abcde", 0, 4, "abcdefghij", 1, 4, -1);
    test("abcde", 0, 4, "abcdefghij", 1, 8, -1);
    test("abcde", 0, 4, "abcdefghij", 1, 9, -1);
    test("abcde", 0, 4, "abcdefghij", 1, 10, -1);
    test("abcde", 0, 4, "abcdefghij", 5, 0, 4);
    test("abcde", 0, 4, "abcdefghij", 5, 1, -5);
    test("abcde", 0, 4, "abcdefghij", 5, 2, -5);
    test("abcde", 0, 4, "abcdefghij", 5, 4, -5);
    test("abcde", 0, 4, "abcdefghij", 5, 5, -5);
    test("abcde", 0, 4, "abcdefghij", 5, 6, -5);
    test("abcde", 0, 4, "abcdefghij", 9, 0, 4);
    test("abcde", 0, 4, "abcdefghij", 9, 1, -9);
    test("abcde", 0, 4, "abcdefghij", 9, 2, -9);
    test("abcde", 0, 4, "abcdefghij", 10, 0, 4);
    test("abcde", 0, 4, "abcdefghij", 10, 1, 4);
    test("abcde", 0, 4, "abcdefghij", 11, 0, 0);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 0, 0, 4);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 0, 1, 3);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 0, 10, -6);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 0, 19, -15);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 0, 20, -16);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 0, 21, -16);
}

void test5()
{
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 1, 0, 4);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 1, 9, -1);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 1, 18, -1);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 1, 19, -1);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 1, 20, -1);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 10, 0, 4);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 10, 1, -10);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 10, 5, -10);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 10, 9, -10);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 19, 0, 4);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 19, 1, -19);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 19, 2, -19);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 20, 0, 4);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 20, 1, 4);
    test("abcde", 0, 4, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 0, 5, "", 0, 0, 5);
    test("abcde", 0, 5, "", 0, 1, 5);
    test("abcde", 0, 5, "", 1, 0, 0);
    test("abcde", 0, 5, "abcde", 0, 0, 5);
    test("abcde", 0, 5, "abcde", 0, 1, 4);
    test("abcde", 0, 5, "abcde", 0, 2, 3);
    test("abcde", 0, 5, "abcde", 0, 4, 1);
    test("abcde", 0, 5, "abcde", 0, 5, 0);
    test("abcde", 0, 5, "abcde", 0, 6, 0);
    test("abcde", 0, 5, "abcde", 1, 0, 5);
    test("abcde", 0, 5, "abcde", 1, 1, -1);
    test("abcde", 0, 5, "abcde", 1, 2, -1);
    test("abcde", 0, 5, "abcde", 1, 3, -1);
    test("abcde", 0, 5, "abcde", 1, 4, -1);
    test("abcde", 0, 5, "abcde", 1, 5, -1);
    test("abcde", 0, 5, "abcde", 2, 0, 5);
    test("abcde", 0, 5, "abcde", 2, 1, -2);
    test("abcde", 0, 5, "abcde", 2, 2, -2);
    test("abcde", 0, 5, "abcde", 2, 3, -2);
    test("abcde", 0, 5, "abcde", 2, 4, -2);
    test("abcde", 0, 5, "abcde", 4, 0, 5);
    test("abcde", 0, 5, "abcde", 4, 1, -4);
    test("abcde", 0, 5, "abcde", 4, 2, -4);
    test("abcde", 0, 5, "abcde", 5, 0, 5);
    test("abcde", 0, 5, "abcde", 5, 1, 5);
    test("abcde", 0, 5, "abcde", 6, 0, 0);
    test("abcde", 0, 5, "abcdefghij", 0, 0, 5);
    test("abcde", 0, 5, "abcdefghij", 0, 1, 4);
    test("abcde", 0, 5, "abcdefghij", 0, 5, 0);
    test("abcde", 0, 5, "abcdefghij", 0, 9, -4);
    test("abcde", 0, 5, "abcdefghij", 0, 10, -5);
    test("abcde", 0, 5, "abcdefghij", 0, 11, -5);
    test("abcde", 0, 5, "abcdefghij", 1, 0, 5);
    test("abcde", 0, 5, "abcdefghij", 1, 1, -1);
    test("abcde", 0, 5, "abcdefghij", 1, 4, -1);
    test("abcde", 0, 5, "abcdefghij", 1, 8, -1);
    test("abcde", 0, 5, "abcdefghij", 1, 9, -1);
    test("abcde", 0, 5, "abcdefghij", 1, 10, -1);
    test("abcde", 0, 5, "abcdefghij", 5, 0, 5);
    test("abcde", 0, 5, "abcdefghij", 5, 1, -5);
    test("abcde", 0, 5, "abcdefghij", 5, 2, -5);
    test("abcde", 0, 5, "abcdefghij", 5, 4, -5);
    test("abcde", 0, 5, "abcdefghij", 5, 5, -5);
    test("abcde", 0, 5, "abcdefghij", 5, 6, -5);
    test("abcde", 0, 5, "abcdefghij", 9, 0, 5);
    test("abcde", 0, 5, "abcdefghij", 9, 1, -9);
    test("abcde", 0, 5, "abcdefghij", 9, 2, -9);
    test("abcde", 0, 5, "abcdefghij", 10, 0, 5);
    test("abcde", 0, 5, "abcdefghij", 10, 1, 5);
    test("abcde", 0, 5, "abcdefghij", 11, 0, 0);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 0, 0, 5);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 0, 1, 4);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 0, 10, -5);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 0, 19, -14);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 0, 20, -15);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 0, 21, -15);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 1, 0, 5);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 1, 9, -1);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 1, 18, -1);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 1, 19, -1);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 1, 20, -1);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 10, 0, 5);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 10, 1, -10);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 10, 5, -10);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 10, 9, -10);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 19, 0, 5);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 19, 1, -19);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 19, 2, -19);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 20, 0, 5);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 20, 1, 5);
    test("abcde", 0, 5, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 0, 6, "", 0, 0, 5);
    test("abcde", 0, 6, "", 0, 1, 5);
    test("abcde", 0, 6, "", 1, 0, 0);
    test("abcde", 0, 6, "abcde", 0, 0, 5);
    test("abcde", 0, 6, "abcde", 0, 1, 4);
    test("abcde", 0, 6, "abcde", 0, 2, 3);
    test("abcde", 0, 6, "abcde", 0, 4, 1);
    test("abcde", 0, 6, "abcde", 0, 5, 0);
}

void test6()
{
    test("abcde", 0, 6, "abcde", 0, 6, 0);
    test("abcde", 0, 6, "abcde", 1, 0, 5);
    test("abcde", 0, 6, "abcde", 1, 1, -1);
    test("abcde", 0, 6, "abcde", 1, 2, -1);
    test("abcde", 0, 6, "abcde", 1, 3, -1);
    test("abcde", 0, 6, "abcde", 1, 4, -1);
    test("abcde", 0, 6, "abcde", 1, 5, -1);
    test("abcde", 0, 6, "abcde", 2, 0, 5);
    test("abcde", 0, 6, "abcde", 2, 1, -2);
    test("abcde", 0, 6, "abcde", 2, 2, -2);
    test("abcde", 0, 6, "abcde", 2, 3, -2);
    test("abcde", 0, 6, "abcde", 2, 4, -2);
    test("abcde", 0, 6, "abcde", 4, 0, 5);
    test("abcde", 0, 6, "abcde", 4, 1, -4);
    test("abcde", 0, 6, "abcde", 4, 2, -4);
    test("abcde", 0, 6, "abcde", 5, 0, 5);
    test("abcde", 0, 6, "abcde", 5, 1, 5);
    test("abcde", 0, 6, "abcde", 6, 0, 0);
    test("abcde", 0, 6, "abcdefghij", 0, 0, 5);
    test("abcde", 0, 6, "abcdefghij", 0, 1, 4);
    test("abcde", 0, 6, "abcdefghij", 0, 5, 0);
    test("abcde", 0, 6, "abcdefghij", 0, 9, -4);
    test("abcde", 0, 6, "abcdefghij", 0, 10, -5);
    test("abcde", 0, 6, "abcdefghij", 0, 11, -5);
    test("abcde", 0, 6, "abcdefghij", 1, 0, 5);
    test("abcde", 0, 6, "abcdefghij", 1, 1, -1);
    test("abcde", 0, 6, "abcdefghij", 1, 4, -1);
    test("abcde", 0, 6, "abcdefghij", 1, 8, -1);
    test("abcde", 0, 6, "abcdefghij", 1, 9, -1);
    test("abcde", 0, 6, "abcdefghij", 1, 10, -1);
    test("abcde", 0, 6, "abcdefghij", 5, 0, 5);
    test("abcde", 0, 6, "abcdefghij", 5, 1, -5);
    test("abcde", 0, 6, "abcdefghij", 5, 2, -5);
    test("abcde", 0, 6, "abcdefghij", 5, 4, -5);
    test("abcde", 0, 6, "abcdefghij", 5, 5, -5);
    test("abcde", 0, 6, "abcdefghij", 5, 6, -5);
    test("abcde", 0, 6, "abcdefghij", 9, 0, 5);
    test("abcde", 0, 6, "abcdefghij", 9, 1, -9);
    test("abcde", 0, 6, "abcdefghij", 9, 2, -9);
    test("abcde", 0, 6, "abcdefghij", 10, 0, 5);
    test("abcde", 0, 6, "abcdefghij", 10, 1, 5);
    test("abcde", 0, 6, "abcdefghij", 11, 0, 0);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 0, 0, 5);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 0, 1, 4);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 0, 10, -5);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 0, 19, -14);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 0, 20, -15);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 0, 21, -15);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 1, 0, 5);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 1, 9, -1);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 1, 18, -1);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 1, 19, -1);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 1, 20, -1);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 10, 0, 5);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 10, 1, -10);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 10, 5, -10);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 10, 9, -10);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 19, 0, 5);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 19, 1, -19);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 19, 2, -19);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 20, 0, 5);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 20, 1, 5);
    test("abcde", 0, 6, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 1, 0, "", 0, 0, 0);
    test("abcde", 1, 0, "", 0, 1, 0);
    test("abcde", 1, 0, "", 1, 0, 0);
    test("abcde", 1, 0, "abcde", 0, 0, 0);
    test("abcde", 1, 0, "abcde", 0, 1, -1);
    test("abcde", 1, 0, "abcde", 0, 2, -2);
    test("abcde", 1, 0, "abcde", 0, 4, -4);
    test("abcde", 1, 0, "abcde", 0, 5, -5);
    test("abcde", 1, 0, "abcde", 0, 6, -5);
    test("abcde", 1, 0, "abcde", 1, 0, 0);
    test("abcde", 1, 0, "abcde", 1, 1, -1);
    test("abcde", 1, 0, "abcde", 1, 2, -2);
    test("abcde", 1, 0, "abcde", 1, 3, -3);
    test("abcde", 1, 0, "abcde", 1, 4, -4);
    test("abcde", 1, 0, "abcde", 1, 5, -4);
    test("abcde", 1, 0, "abcde", 2, 0, 0);
    test("abcde", 1, 0, "abcde", 2, 1, -1);
    test("abcde", 1, 0, "abcde", 2, 2, -2);
    test("abcde", 1, 0, "abcde", 2, 3, -3);
    test("abcde", 1, 0, "abcde", 2, 4, -3);
    test("abcde", 1, 0, "abcde", 4, 0, 0);
    test("abcde", 1, 0, "abcde", 4, 1, -1);
    test("abcde", 1, 0, "abcde", 4, 2, -1);
    test("abcde", 1, 0, "abcde", 5, 0, 0);
    test("abcde", 1, 0, "abcde", 5, 1, 0);
    test("abcde", 1, 0, "abcde", 6, 0, 0);
    test("abcde", 1, 0, "abcdefghij", 0, 0, 0);
    test("abcde", 1, 0, "abcdefghij", 0, 1, -1);
    test("abcde", 1, 0, "abcdefghij", 0, 5, -5);
    test("abcde", 1, 0, "abcdefghij", 0, 9, -9);
    test("abcde", 1, 0, "abcdefghij", 0, 10, -10);
    test("abcde", 1, 0, "abcdefghij", 0, 11, -10);
    test("abcde", 1, 0, "abcdefghij", 1, 0, 0);
    test("abcde", 1, 0, "abcdefghij", 1, 1, -1);
}

void test7()
{
    test("abcde", 1, 0, "abcdefghij", 1, 4, -4);
    test("abcde", 1, 0, "abcdefghij", 1, 8, -8);
    test("abcde", 1, 0, "abcdefghij", 1, 9, -9);
    test("abcde", 1, 0, "abcdefghij", 1, 10, -9);
    test("abcde", 1, 0, "abcdefghij", 5, 0, 0);
    test("abcde", 1, 0, "abcdefghij", 5, 1, -1);
    test("abcde", 1, 0, "abcdefghij", 5, 2, -2);
    test("abcde", 1, 0, "abcdefghij", 5, 4, -4);
    test("abcde", 1, 0, "abcdefghij", 5, 5, -5);
    test("abcde", 1, 0, "abcdefghij", 5, 6, -5);
    test("abcde", 1, 0, "abcdefghij", 9, 0, 0);
    test("abcde", 1, 0, "abcdefghij", 9, 1, -1);
    test("abcde", 1, 0, "abcdefghij", 9, 2, -1);
    test("abcde", 1, 0, "abcdefghij", 10, 0, 0);
    test("abcde", 1, 0, "abcdefghij", 10, 1, 0);
    test("abcde", 1, 0, "abcdefghij", 11, 0, 0);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 0, 0, 0);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 0, 1, -1);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 0, 10, -10);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 0, 19, -19);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 0, 20, -20);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 0, 21, -20);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 1, 0, 0);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 1, 9, -9);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 1, 18, -18);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 1, 19, -19);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 1, 20, -19);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 10, 0, 0);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 10, 1, -1);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 19, 0, 0);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 19, 1, -1);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 19, 2, -1);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 20, 0, 0);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 20, 1, 0);
    test("abcde", 1, 0, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 1, 1, "", 0, 0, 1);
    test("abcde", 1, 1, "", 0, 1, 1);
    test("abcde", 1, 1, "", 1, 0, 0);
    test("abcde", 1, 1, "abcde", 0, 0, 1);
    test("abcde", 1, 1, "abcde", 0, 1, 1);
    test("abcde", 1, 1, "abcde", 0, 2, 1);
    test("abcde", 1, 1, "abcde", 0, 4, 1);
    test("abcde", 1, 1, "abcde", 0, 5, 1);
    test("abcde", 1, 1, "abcde", 0, 6, 1);
    test("abcde", 1, 1, "abcde", 1, 0, 1);
    test("abcde", 1, 1, "abcde", 1, 1, 0);
    test("abcde", 1, 1, "abcde", 1, 2, -1);
    test("abcde", 1, 1, "abcde", 1, 3, -2);
    test("abcde", 1, 1, "abcde", 1, 4, -3);
    test("abcde", 1, 1, "abcde", 1, 5, -3);
    test("abcde", 1, 1, "abcde", 2, 0, 1);
    test("abcde", 1, 1, "abcde", 2, 1, -1);
    test("abcde", 1, 1, "abcde", 2, 2, -1);
    test("abcde", 1, 1, "abcde", 2, 3, -1);
    test("abcde", 1, 1, "abcde", 2, 4, -1);
    test("abcde", 1, 1, "abcde", 4, 0, 1);
    test("abcde", 1, 1, "abcde", 4, 1, -3);
    test("abcde", 1, 1, "abcde", 4, 2, -3);
    test("abcde", 1, 1, "abcde", 5, 0, 1);
    test("abcde", 1, 1, "abcde", 5, 1, 1);
    test("abcde", 1, 1, "abcde", 6, 0, 0);
    test("abcde", 1, 1, "abcdefghij", 0, 0, 1);
    test("abcde", 1, 1, "abcdefghij", 0, 1, 1);
    test("abcde", 1, 1, "abcdefghij", 0, 5, 1);
    test("abcde", 1, 1, "abcdefghij", 0, 9, 1);
    test("abcde", 1, 1, "abcdefghij", 0, 10, 1);
    test("abcde", 1, 1, "abcdefghij", 0, 11, 1);
    test("abcde", 1, 1, "abcdefghij", 1, 0, 1);
    test("abcde", 1, 1, "abcdefghij", 1, 1, 0);
    test("abcde", 1, 1, "abcdefghij", 1, 4, -3);
    test("abcde", 1, 1, "abcdefghij", 1, 8, -7);
    test("abcde", 1, 1, "abcdefghij", 1, 9, -8);
    test("abcde", 1, 1, "abcdefghij", 1, 10, -8);
    test("abcde", 1, 1, "abcdefghij", 5, 0, 1);
    test("abcde", 1, 1, "abcdefghij", 5, 1, -4);
    test("abcde", 1, 1, "abcdefghij", 5, 2, -4);
    test("abcde", 1, 1, "abcdefghij", 5, 4, -4);
    test("abcde", 1, 1, "abcdefghij", 5, 5, -4);
    test("abcde", 1, 1, "abcdefghij", 5, 6, -4);
    test("abcde", 1, 1, "abcdefghij", 9, 0, 1);
    test("abcde", 1, 1, "abcdefghij", 9, 1, -8);
    test("abcde", 1, 1, "abcdefghij", 9, 2, -8);
    test("abcde", 1, 1, "abcdefghij", 10, 0, 1);
    test("abcde", 1, 1, "abcdefghij", 10, 1, 1);
    test("abcde", 1, 1, "abcdefghij", 11, 0, 0);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 0, 0, 1);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 0, 1, 1);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 0, 10, 1);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 0, 19, 1);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 0, 20, 1);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 0, 21, 1);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 1, 0, 1);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 1, 1, 0);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 1, 9, -8);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 1, 18, -17);
}

void test8()
{
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 1, 19, -18);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 1, 20, -18);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 10, 0, 1);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 10, 1, -9);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 10, 5, -9);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 10, 10, -9);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 10, 11, -9);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 19, 0, 1);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 19, 1, -18);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 19, 2, -18);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 20, 0, 1);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 20, 1, 1);
    test("abcde", 1, 1, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 1, 2, "", 0, 0, 2);
    test("abcde", 1, 2, "", 0, 1, 2);
    test("abcde", 1, 2, "", 1, 0, 0);
    test("abcde", 1, 2, "abcde", 0, 0, 2);
    test("abcde", 1, 2, "abcde", 0, 1, 1);
    test("abcde", 1, 2, "abcde", 0, 2, 1);
    test("abcde", 1, 2, "abcde", 0, 4, 1);
    test("abcde", 1, 2, "abcde", 0, 5, 1);
    test("abcde", 1, 2, "abcde", 0, 6, 1);
    test("abcde", 1, 2, "abcde", 1, 0, 2);
    test("abcde", 1, 2, "abcde", 1, 1, 1);
    test("abcde", 1, 2, "abcde", 1, 2, 0);
    test("abcde", 1, 2, "abcde", 1, 3, -1);
    test("abcde", 1, 2, "abcde", 1, 4, -2);
    test("abcde", 1, 2, "abcde", 1, 5, -2);
    test("abcde", 1, 2, "abcde", 2, 0, 2);
    test("abcde", 1, 2, "abcde", 2, 1, -1);
    test("abcde", 1, 2, "abcde", 2, 2, -1);
    test("abcde", 1, 2, "abcde", 2, 3, -1);
    test("abcde", 1, 2, "abcde", 2, 4, -1);
    test("abcde", 1, 2, "abcde", 4, 0, 2);
    test("abcde", 1, 2, "abcde", 4, 1, -3);
    test("abcde", 1, 2, "abcde", 4, 2, -3);
    test("abcde", 1, 2, "abcde", 5, 0, 2);
    test("abcde", 1, 2, "abcde", 5, 1, 2);
    test("abcde", 1, 2, "abcde", 6, 0, 0);
    test("abcde", 1, 2, "abcdefghij", 0, 0, 2);
    test("abcde", 1, 2, "abcdefghij", 0, 1, 1);
    test("abcde", 1, 2, "abcdefghij", 0, 5, 1);
    test("abcde", 1, 2, "abcdefghij", 0, 9, 1);
    test("abcde", 1, 2, "abcdefghij", 0, 10, 1);
    test("abcde", 1, 2, "abcdefghij", 0, 11, 1);
    test("abcde", 1, 2, "abcdefghij", 1, 0, 2);
    test("abcde", 1, 2, "abcdefghij", 1, 1, 1);
    test("abcde", 1, 2, "abcdefghij", 1, 4, -2);
    test("abcde", 1, 2, "abcdefghij", 1, 8, -6);
    test("abcde", 1, 2, "abcdefghij", 1, 9, -7);
    test("abcde", 1, 2, "abcdefghij", 1, 10, -7);
    test("abcde", 1, 2, "abcdefghij", 5, 0, 2);
    test("abcde", 1, 2, "abcdefghij", 5, 1, -4);
    test("abcde", 1, 2, "abcdefghij", 5, 2, -4);
    test("abcde", 1, 2, "abcdefghij", 5, 4, -4);
    test("abcde", 1, 2, "abcdefghij", 5, 5, -4);
    test("abcde", 1, 2, "abcdefghij", 5, 6, -4);
    test("abcde", 1, 2, "abcdefghij", 9, 0, 2);
    test("abcde", 1, 2, "abcdefghij", 9, 1, -8);
    test("abcde", 1, 2, "abcdefghij", 9, 2, -8);
    test("abcde", 1, 2, "abcdefghij", 10, 0, 2);
    test("abcde", 1, 2, "abcdefghij", 10, 1, 2);
    test("abcde", 1, 2, "abcdefghij", 11, 0, 0);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 0, 0, 2);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 0, 1, 1);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 0, 10, 1);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 0, 19, 1);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 0, 20, 1);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 0, 21, 1);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 1, 0, 2);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 1, 1, 1);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 1, 9, -7);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 1, 18, -16);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 1, 19, -17);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 1, 20, -17);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 10, 0, 2);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 10, 1, -9);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 10, 5, -9);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 10, 10, -9);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 10, 11, -9);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 19, 0, 2);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 19, 1, -18);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 19, 2, -18);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 20, 0, 2);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 20, 1, 2);
    test("abcde", 1, 2, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 1, 3, "", 0, 0, 3);
    test("abcde", 1, 3, "", 0, 1, 3);
    test("abcde", 1, 3, "", 1, 0, 0);
    test("abcde", 1, 3, "abcde", 0, 0, 3);
    test("abcde", 1, 3, "abcde", 0, 1, 1);
    test("abcde", 1, 3, "abcde", 0, 2, 1);
    test("abcde", 1, 3, "abcde", 0, 4, 1);
    test("abcde", 1, 3, "abcde", 0, 5, 1);
    test("abcde", 1, 3, "abcde", 0, 6, 1);
    test("abcde", 1, 3, "abcde", 1, 0, 3);
    test("abcde", 1, 3, "abcde", 1, 1, 2);
    test("abcde", 1, 3, "abcde", 1, 2, 1);
}

void test9()
{
    test("abcde", 1, 3, "abcde", 1, 3, 0);
    test("abcde", 1, 3, "abcde", 1, 4, -1);
    test("abcde", 1, 3, "abcde", 1, 5, -1);
    test("abcde", 1, 3, "abcde", 2, 0, 3);
    test("abcde", 1, 3, "abcde", 2, 1, -1);
    test("abcde", 1, 3, "abcde", 2, 2, -1);
    test("abcde", 1, 3, "abcde", 2, 3, -1);
    test("abcde", 1, 3, "abcde", 2, 4, -1);
    test("abcde", 1, 3, "abcde", 4, 0, 3);
    test("abcde", 1, 3, "abcde", 4, 1, -3);
    test("abcde", 1, 3, "abcde", 4, 2, -3);
    test("abcde", 1, 3, "abcde", 5, 0, 3);
    test("abcde", 1, 3, "abcde", 5, 1, 3);
    test("abcde", 1, 3, "abcde", 6, 0, 0);
    test("abcde", 1, 3, "abcdefghij", 0, 0, 3);
    test("abcde", 1, 3, "abcdefghij", 0, 1, 1);
    test("abcde", 1, 3, "abcdefghij", 0, 5, 1);
    test("abcde", 1, 3, "abcdefghij", 0, 9, 1);
    test("abcde", 1, 3, "abcdefghij", 0, 10, 1);
    test("abcde", 1, 3, "abcdefghij", 0, 11, 1);
    test("abcde", 1, 3, "abcdefghij", 1, 0, 3);
    test("abcde", 1, 3, "abcdefghij", 1, 1, 2);
    test("abcde", 1, 3, "abcdefghij", 1, 4, -1);
    test("abcde", 1, 3, "abcdefghij", 1, 8, -5);
    test("abcde", 1, 3, "abcdefghij", 1, 9, -6);
    test("abcde", 1, 3, "abcdefghij", 1, 10, -6);
    test("abcde", 1, 3, "abcdefghij", 5, 0, 3);
    test("abcde", 1, 3, "abcdefghij", 5, 1, -4);
    test("abcde", 1, 3, "abcdefghij", 5, 2, -4);
    test("abcde", 1, 3, "abcdefghij", 5, 4, -4);
    test("abcde", 1, 3, "abcdefghij", 5, 5, -4);
    test("abcde", 1, 3, "abcdefghij", 5, 6, -4);
    test("abcde", 1, 3, "abcdefghij", 9, 0, 3);
    test("abcde", 1, 3, "abcdefghij", 9, 1, -8);
    test("abcde", 1, 3, "abcdefghij", 9, 2, -8);
    test("abcde", 1, 3, "abcdefghij", 10, 0, 3);
    test("abcde", 1, 3, "abcdefghij", 10, 1, 3);
    test("abcde", 1, 3, "abcdefghij", 11, 0, 0);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 0, 0, 3);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 0, 1, 1);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 0, 10, 1);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 0, 19, 1);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 0, 20, 1);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 0, 21, 1);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 1, 0, 3);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 1, 1, 2);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 1, 9, -6);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 1, 18, -15);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 1, 19, -16);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 1, 20, -16);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 10, 0, 3);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 10, 1, -9);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 10, 5, -9);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 10, 10, -9);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 10, 11, -9);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 19, 0, 3);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 19, 1, -18);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 19, 2, -18);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 20, 0, 3);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 20, 1, 3);
    test("abcde", 1, 3, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 1, 4, "", 0, 0, 4);
    test("abcde", 1, 4, "", 0, 1, 4);
    test("abcde", 1, 4, "", 1, 0, 0);
    test("abcde", 1, 4, "abcde", 0, 0, 4);
    test("abcde", 1, 4, "abcde", 0, 1, 1);
    test("abcde", 1, 4, "abcde", 0, 2, 1);
    test("abcde", 1, 4, "abcde", 0, 4, 1);
    test("abcde", 1, 4, "abcde", 0, 5, 1);
    test("abcde", 1, 4, "abcde", 0, 6, 1);
    test("abcde", 1, 4, "abcde", 1, 0, 4);
    test("abcde", 1, 4, "abcde", 1, 1, 3);
    test("abcde", 1, 4, "abcde", 1, 2, 2);
    test("abcde", 1, 4, "abcde", 1, 3, 1);
    test("abcde", 1, 4, "abcde", 1, 4, 0);
    test("abcde", 1, 4, "abcde", 1, 5, 0);
    test("abcde", 1, 4, "abcde", 2, 0, 4);
    test("abcde", 1, 4, "abcde", 2, 1, -1);
    test("abcde", 1, 4, "abcde", 2, 2, -1);
    test("abcde", 1, 4, "abcde", 2, 3, -1);
    test("abcde", 1, 4, "abcde", 2, 4, -1);
    test("abcde", 1, 4, "abcde", 4, 0, 4);
    test("abcde", 1, 4, "abcde", 4, 1, -3);
    test("abcde", 1, 4, "abcde", 4, 2, -3);
    test("abcde", 1, 4, "abcde", 5, 0, 4);
    test("abcde", 1, 4, "abcde", 5, 1, 4);
    test("abcde", 1, 4, "abcde", 6, 0, 0);
    test("abcde", 1, 4, "abcdefghij", 0, 0, 4);
    test("abcde", 1, 4, "abcdefghij", 0, 1, 1);
    test("abcde", 1, 4, "abcdefghij", 0, 5, 1);
    test("abcde", 1, 4, "abcdefghij", 0, 9, 1);
    test("abcde", 1, 4, "abcdefghij", 0, 10, 1);
    test("abcde", 1, 4, "abcdefghij", 0, 11, 1);
    test("abcde", 1, 4, "abcdefghij", 1, 0, 4);
    test("abcde", 1, 4, "abcdefghij", 1, 1, 3);
    test("abcde", 1, 4, "abcdefghij", 1, 4, 0);
    test("abcde", 1, 4, "abcdefghij", 1, 8, -4);
    test("abcde", 1, 4, "abcdefghij", 1, 9, -5);
    test("abcde", 1, 4, "abcdefghij", 1, 10, -5);
}

void test10()
{
    test("abcde", 1, 4, "abcdefghij", 5, 0, 4);
    test("abcde", 1, 4, "abcdefghij", 5, 1, -4);
    test("abcde", 1, 4, "abcdefghij", 5, 2, -4);
    test("abcde", 1, 4, "abcdefghij", 5, 4, -4);
    test("abcde", 1, 4, "abcdefghij", 5, 5, -4);
    test("abcde", 1, 4, "abcdefghij", 5, 6, -4);
    test("abcde", 1, 4, "abcdefghij", 9, 0, 4);
    test("abcde", 1, 4, "abcdefghij", 9, 1, -8);
    test("abcde", 1, 4, "abcdefghij", 9, 2, -8);
    test("abcde", 1, 4, "abcdefghij", 10, 0, 4);
    test("abcde", 1, 4, "abcdefghij", 10, 1, 4);
    test("abcde", 1, 4, "abcdefghij", 11, 0, 0);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 0, 0, 4);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 0, 1, 1);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 0, 10, 1);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 0, 19, 1);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 0, 20, 1);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 0, 21, 1);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 1, 0, 4);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 1, 1, 3);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 1, 9, -5);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 1, 18, -14);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 1, 19, -15);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 1, 20, -15);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 10, 0, 4);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 10, 1, -9);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 10, 5, -9);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 10, 10, -9);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 10, 11, -9);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 19, 0, 4);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 19, 1, -18);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 19, 2, -18);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 20, 0, 4);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 20, 1, 4);
    test("abcde", 1, 4, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 1, 5, "", 0, 0, 4);
    test("abcde", 1, 5, "", 0, 1, 4);
    test("abcde", 1, 5, "", 1, 0, 0);
    test("abcde", 1, 5, "abcde", 0, 0, 4);
    test("abcde", 1, 5, "abcde", 0, 1, 1);
    test("abcde", 1, 5, "abcde", 0, 2, 1);
    test("abcde", 1, 5, "abcde", 0, 4, 1);
    test("abcde", 1, 5, "abcde", 0, 5, 1);
    test("abcde", 1, 5, "abcde", 0, 6, 1);
    test("abcde", 1, 5, "abcde", 1, 0, 4);
    test("abcde", 1, 5, "abcde", 1, 1, 3);
    test("abcde", 1, 5, "abcde", 1, 2, 2);
    test("abcde", 1, 5, "abcde", 1, 3, 1);
    test("abcde", 1, 5, "abcde", 1, 4, 0);
    test("abcde", 1, 5, "abcde", 1, 5, 0);
    test("abcde", 1, 5, "abcde", 2, 0, 4);
    test("abcde", 1, 5, "abcde", 2, 1, -1);
    test("abcde", 1, 5, "abcde", 2, 2, -1);
    test("abcde", 1, 5, "abcde", 2, 3, -1);
    test("abcde", 1, 5, "abcde", 2, 4, -1);
    test("abcde", 1, 5, "abcde", 4, 0, 4);
    test("abcde", 1, 5, "abcde", 4, 1, -3);
    test("abcde", 1, 5, "abcde", 4, 2, -3);
    test("abcde", 1, 5, "abcde", 5, 0, 4);
    test("abcde", 1, 5, "abcde", 5, 1, 4);
    test("abcde", 1, 5, "abcde", 6, 0, 0);
    test("abcde", 1, 5, "abcdefghij", 0, 0, 4);
    test("abcde", 1, 5, "abcdefghij", 0, 1, 1);
    test("abcde", 1, 5, "abcdefghij", 0, 5, 1);
    test("abcde", 1, 5, "abcdefghij", 0, 9, 1);
    test("abcde", 1, 5, "abcdefghij", 0, 10, 1);
    test("abcde", 1, 5, "abcdefghij", 0, 11, 1);
    test("abcde", 1, 5, "abcdefghij", 1, 0, 4);
    test("abcde", 1, 5, "abcdefghij", 1, 1, 3);
    test("abcde", 1, 5, "abcdefghij", 1, 4, 0);
    test("abcde", 1, 5, "abcdefghij", 1, 8, -4);
    test("abcde", 1, 5, "abcdefghij", 1, 9, -5);
    test("abcde", 1, 5, "abcdefghij", 1, 10, -5);
    test("abcde", 1, 5, "abcdefghij", 5, 0, 4);
    test("abcde", 1, 5, "abcdefghij", 5, 1, -4);
    test("abcde", 1, 5, "abcdefghij", 5, 2, -4);
    test("abcde", 1, 5, "abcdefghij", 5, 4, -4);
    test("abcde", 1, 5, "abcdefghij", 5, 5, -4);
    test("abcde", 1, 5, "abcdefghij", 5, 6, -4);
    test("abcde", 1, 5, "abcdefghij", 9, 0, 4);
    test("abcde", 1, 5, "abcdefghij", 9, 1, -8);
    test("abcde", 1, 5, "abcdefghij", 9, 2, -8);
    test("abcde", 1, 5, "abcdefghij", 10, 0, 4);
    test("abcde", 1, 5, "abcdefghij", 10, 1, 4);
    test("abcde", 1, 5, "abcdefghij", 11, 0, 0);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 0, 0, 4);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 0, 1, 1);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 0, 10, 1);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 0, 19, 1);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 0, 20, 1);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 0, 21, 1);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 1, 0, 4);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 1, 1, 3);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 1, 9, -5);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 1, 18, -14);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 1, 19, -15);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 1, 20, -15);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 10, 0, 4);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 10, 1, -9);
}

void test11()
{
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 10, 5, -9);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 10, 10, -9);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 10, 11, -9);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 19, 0, 4);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 19, 1, -18);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 19, 2, -18);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 20, 0, 4);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 20, 1, 4);
    test("abcde", 1, 5, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 2, 0, "", 0, 0, 0);
    test("abcde", 2, 0, "", 0, 1, 0);
    test("abcde", 2, 0, "", 1, 0, 0);
    test("abcde", 2, 0, "abcde", 0, 0, 0);
    test("abcde", 2, 0, "abcde", 0, 1, -1);
    test("abcde", 2, 0, "abcde", 0, 2, -2);
    test("abcde", 2, 0, "abcde", 0, 4, -4);
    test("abcde", 2, 0, "abcde", 0, 5, -5);
    test("abcde", 2, 0, "abcde", 0, 6, -5);
    test("abcde", 2, 0, "abcde", 1, 0, 0);
    test("abcde", 2, 0, "abcde", 1, 1, -1);
    test("abcde", 2, 0, "abcde", 1, 2, -2);
    test("abcde", 2, 0, "abcde", 1, 3, -3);
    test("abcde", 2, 0, "abcde", 1, 4, -4);
    test("abcde", 2, 0, "abcde", 1, 5, -4);
    test("abcde", 2, 0, "abcde", 2, 0, 0);
    test("abcde", 2, 0, "abcde", 2, 1, -1);
    test("abcde", 2, 0, "abcde", 2, 2, -2);
    test("abcde", 2, 0, "abcde", 2, 3, -3);
    test("abcde", 2, 0, "abcde", 2, 4, -3);
    test("abcde", 2, 0, "abcde", 4, 0, 0);
    test("abcde", 2, 0, "abcde", 4, 1, -1);
    test("abcde", 2, 0, "abcde", 4, 2, -1);
    test("abcde", 2, 0, "abcde", 5, 0, 0);
    test("abcde", 2, 0, "abcde", 5, 1, 0);
    test("abcde", 2, 0, "abcde", 6, 0, 0);
    test("abcde", 2, 0, "abcdefghij", 0, 0, 0);
    test("abcde", 2, 0, "abcdefghij", 0, 1, -1);
    test("abcde", 2, 0, "abcdefghij", 0, 5, -5);
    test("abcde", 2, 0, "abcdefghij", 0, 9, -9);
    test("abcde", 2, 0, "abcdefghij", 0, 10, -10);
    test("abcde", 2, 0, "abcdefghij", 0, 11, -10);
    test("abcde", 2, 0, "abcdefghij", 1, 0, 0);
    test("abcde", 2, 0, "abcdefghij", 1, 1, -1);
    test("abcde", 2, 0, "abcdefghij", 1, 4, -4);
    test("abcde", 2, 0, "abcdefghij", 1, 8, -8);
    test("abcde", 2, 0, "abcdefghij", 1, 9, -9);
    test("abcde", 2, 0, "abcdefghij", 1, 10, -9);
    test("abcde", 2, 0, "abcdefghij", 5, 0, 0);
    test("abcde", 2, 0, "abcdefghij", 5, 1, -1);
    test("abcde", 2, 0, "abcdefghij", 5, 2, -2);
    test("abcde", 2, 0, "abcdefghij", 5, 4, -4);
    test("abcde", 2, 0, "abcdefghij", 5, 5, -5);
    test("abcde", 2, 0, "abcdefghij", 5, 6, -5);
    test("abcde", 2, 0, "abcdefghij", 9, 0, 0);
    test("abcde", 2, 0, "abcdefghij", 9, 1, -1);
    test("abcde", 2, 0, "abcdefghij", 9, 2, -1);
    test("abcde", 2, 0, "abcdefghij", 10, 0, 0);
    test("abcde", 2, 0, "abcdefghij", 10, 1, 0);
    test("abcde", 2, 0, "abcdefghij", 11, 0, 0);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 0, 0, 0);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 0, 1, -1);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 0, 10, -10);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 0, 19, -19);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 0, 20, -20);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 0, 21, -20);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 1, 0, 0);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 1, 9, -9);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 1, 18, -18);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 1, 19, -19);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 1, 20, -19);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 10, 0, 0);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 10, 1, -1);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 19, 0, 0);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 19, 1, -1);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 19, 2, -1);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 20, 0, 0);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 20, 1, 0);
    test("abcde", 2, 0, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 2, 1, "", 0, 0, 1);
    test("abcde", 2, 1, "", 0, 1, 1);
    test("abcde", 2, 1, "", 1, 0, 0);
    test("abcde", 2, 1, "abcde", 0, 0, 1);
    test("abcde", 2, 1, "abcde", 0, 1, 2);
    test("abcde", 2, 1, "abcde", 0, 2, 2);
    test("abcde", 2, 1, "abcde", 0, 4, 2);
    test("abcde", 2, 1, "abcde", 0, 5, 2);
    test("abcde", 2, 1, "abcde", 0, 6, 2);
    test("abcde", 2, 1, "abcde", 1, 0, 1);
    test("abcde", 2, 1, "abcde", 1, 1, 1);
    test("abcde", 2, 1, "abcde", 1, 2, 1);
    test("abcde", 2, 1, "abcde", 1, 3, 1);
    test("abcde", 2, 1, "abcde", 1, 4, 1);
    test("abcde", 2, 1, "abcde", 1, 5, 1);
    test("abcde", 2, 1, "abcde", 2, 0, 1);
}

void test12()
{
    test("abcde", 2, 1, "abcde", 2, 1, 0);
    test("abcde", 2, 1, "abcde", 2, 2, -1);
    test("abcde", 2, 1, "abcde", 2, 3, -2);
    test("abcde", 2, 1, "abcde", 2, 4, -2);
    test("abcde", 2, 1, "abcde", 4, 0, 1);
    test("abcde", 2, 1, "abcde", 4, 1, -2);
    test("abcde", 2, 1, "abcde", 4, 2, -2);
    test("abcde", 2, 1, "abcde", 5, 0, 1);
    test("abcde", 2, 1, "abcde", 5, 1, 1);
    test("abcde", 2, 1, "abcde", 6, 0, 0);
    test("abcde", 2, 1, "abcdefghij", 0, 0, 1);
    test("abcde", 2, 1, "abcdefghij", 0, 1, 2);
    test("abcde", 2, 1, "abcdefghij", 0, 5, 2);
    test("abcde", 2, 1, "abcdefghij", 0, 9, 2);
    test("abcde", 2, 1, "abcdefghij", 0, 10, 2);
    test("abcde", 2, 1, "abcdefghij", 0, 11, 2);
    test("abcde", 2, 1, "abcdefghij", 1, 0, 1);
    test("abcde", 2, 1, "abcdefghij", 1, 1, 1);
    test("abcde", 2, 1, "abcdefghij", 1, 4, 1);
    test("abcde", 2, 1, "abcdefghij", 1, 8, 1);
    test("abcde", 2, 1, "abcdefghij", 1, 9, 1);
    test("abcde", 2, 1, "abcdefghij", 1, 10, 1);
    test("abcde", 2, 1, "abcdefghij", 5, 0, 1);
    test("abcde", 2, 1, "abcdefghij", 5, 1, -3);
    test("abcde", 2, 1, "abcdefghij", 5, 2, -3);
    test("abcde", 2, 1, "abcdefghij", 5, 4, -3);
    test("abcde", 2, 1, "abcdefghij", 5, 5, -3);
    test("abcde", 2, 1, "abcdefghij", 5, 6, -3);
    test("abcde", 2, 1, "abcdefghij", 9, 0, 1);
    test("abcde", 2, 1, "abcdefghij", 9, 1, -7);
    test("abcde", 2, 1, "abcdefghij", 9, 2, -7);
    test("abcde", 2, 1, "abcdefghij", 10, 0, 1);
    test("abcde", 2, 1, "abcdefghij", 10, 1, 1);
    test("abcde", 2, 1, "abcdefghij", 11, 0, 0);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 0, 0, 1);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 0, 1, 2);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 0, 10, 2);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 0, 19, 2);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 0, 20, 2);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 0, 21, 2);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 1, 0, 1);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 1, 1, 1);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 1, 9, 1);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 1, 18, 1);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 1, 19, 1);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 1, 20, 1);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 10, 0, 1);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 10, 1, -8);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 10, 5, -8);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 10, 9, -8);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 10, 10, -8);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 10, 11, -8);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 19, 0, 1);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 19, 1, -17);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 19, 2, -17);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 20, 0, 1);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 20, 1, 1);
    test("abcde", 2, 1, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 2, 2, "", 0, 0, 2);
    test("abcde", 2, 2, "", 0, 1, 2);
    test("abcde", 2, 2, "", 1, 0, 0);
    test("abcde", 2, 2, "abcde", 0, 0, 2);
    test("abcde", 2, 2, "abcde", 0, 1, 2);
    test("abcde", 2, 2, "abcde", 0, 2, 2);
    test("abcde", 2, 2, "abcde", 0, 4, 2);
    test("abcde", 2, 2, "abcde", 0, 5, 2);
    test("abcde", 2, 2, "abcde", 0, 6, 2);
    test("abcde", 2, 2, "abcde", 1, 0, 2);
    test("abcde", 2, 2, "abcde", 1, 1, 1);
    test("abcde", 2, 2, "abcde", 1, 2, 1);
    test("abcde", 2, 2, "abcde", 1, 3, 1);
    test("abcde", 2, 2, "abcde", 1, 4, 1);
    test("abcde", 2, 2, "abcde", 1, 5, 1);
    test("abcde", 2, 2, "abcde", 2, 0, 2);
    test("abcde", 2, 2, "abcde", 2, 1, 1);
    test("abcde", 2, 2, "abcde", 2, 2, 0);
    test("abcde", 2, 2, "abcde", 2, 3, -1);
    test("abcde", 2, 2, "abcde", 2, 4, -1);
    test("abcde", 2, 2, "abcde", 4, 0, 2);
    test("abcde", 2, 2, "abcde", 4, 1, -2);
    test("abcde", 2, 2, "abcde", 4, 2, -2);
    test("abcde", 2, 2, "abcde", 5, 0, 2);
    test("abcde", 2, 2, "abcde", 5, 1, 2);
    test("abcde", 2, 2, "abcde", 6, 0, 0);
    test("abcde", 2, 2, "abcdefghij", 0, 0, 2);
    test("abcde", 2, 2, "abcdefghij", 0, 1, 2);
    test("abcde", 2, 2, "abcdefghij", 0, 5, 2);
    test("abcde", 2, 2, "abcdefghij", 0, 9, 2);
    test("abcde", 2, 2, "abcdefghij", 0, 10, 2);
    test("abcde", 2, 2, "abcdefghij", 0, 11, 2);
    test("abcde", 2, 2, "abcdefghij", 1, 0, 2);
    test("abcde", 2, 2, "abcdefghij", 1, 1, 1);
    test("abcde", 2, 2, "abcdefghij", 1, 4, 1);
    test("abcde", 2, 2, "abcdefghij", 1, 8, 1);
    test("abcde", 2, 2, "abcdefghij", 1, 9, 1);
    test("abcde", 2, 2, "abcdefghij", 1, 10, 1);
    test("abcde", 2, 2, "abcdefghij", 5, 0, 2);
    test("abcde", 2, 2, "abcdefghij", 5, 1, -3);
    test("abcde", 2, 2, "abcdefghij", 5, 2, -3);
    test("abcde", 2, 2, "abcdefghij", 5, 4, -3);
}

void test13()
{
    test("abcde", 2, 2, "abcdefghij", 5, 5, -3);
    test("abcde", 2, 2, "abcdefghij", 5, 6, -3);
    test("abcde", 2, 2, "abcdefghij", 9, 0, 2);
    test("abcde", 2, 2, "abcdefghij", 9, 1, -7);
    test("abcde", 2, 2, "abcdefghij", 9, 2, -7);
    test("abcde", 2, 2, "abcdefghij", 10, 0, 2);
    test("abcde", 2, 2, "abcdefghij", 10, 1, 2);
    test("abcde", 2, 2, "abcdefghij", 11, 0, 0);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 0, 0, 2);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 0, 1, 2);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 0, 10, 2);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 0, 19, 2);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 0, 20, 2);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 0, 21, 2);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 1, 0, 2);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 1, 1, 1);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 1, 9, 1);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 1, 18, 1);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 1, 19, 1);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 1, 20, 1);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 10, 0, 2);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 10, 1, -8);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 10, 5, -8);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 10, 9, -8);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 10, 10, -8);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 10, 11, -8);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 19, 0, 2);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 19, 1, -17);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 19, 2, -17);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 20, 0, 2);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 20, 1, 2);
    test("abcde", 2, 2, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 2, 3, "", 0, 0, 3);
    test("abcde", 2, 3, "", 0, 1, 3);
    test("abcde", 2, 3, "", 1, 0, 0);
    test("abcde", 2, 3, "abcde", 0, 0, 3);
    test("abcde", 2, 3, "abcde", 0, 1, 2);
    test("abcde", 2, 3, "abcde", 0, 2, 2);
    test("abcde", 2, 3, "abcde", 0, 4, 2);
    test("abcde", 2, 3, "abcde", 0, 5, 2);
    test("abcde", 2, 3, "abcde", 0, 6, 2);
    test("abcde", 2, 3, "abcde", 1, 0, 3);
    test("abcde", 2, 3, "abcde", 1, 1, 1);
    test("abcde", 2, 3, "abcde", 1, 2, 1);
    test("abcde", 2, 3, "abcde", 1, 3, 1);
    test("abcde", 2, 3, "abcde", 1, 4, 1);
    test("abcde", 2, 3, "abcde", 1, 5, 1);
    test("abcde", 2, 3, "abcde", 2, 0, 3);
    test("abcde", 2, 3, "abcde", 2, 1, 2);
    test("abcde", 2, 3, "abcde", 2, 2, 1);
    test("abcde", 2, 3, "abcde", 2, 3, 0);
    test("abcde", 2, 3, "abcde", 2, 4, 0);
    test("abcde", 2, 3, "abcde", 4, 0, 3);
    test("abcde", 2, 3, "abcde", 4, 1, -2);
    test("abcde", 2, 3, "abcde", 4, 2, -2);
    test("abcde", 2, 3, "abcde", 5, 0, 3);
    test("abcde", 2, 3, "abcde", 5, 1, 3);
    test("abcde", 2, 3, "abcde", 6, 0, 0);
    test("abcde", 2, 3, "abcdefghij", 0, 0, 3);
    test("abcde", 2, 3, "abcdefghij", 0, 1, 2);
    test("abcde", 2, 3, "abcdefghij", 0, 5, 2);
    test("abcde", 2, 3, "abcdefghij", 0, 9, 2);
    test("abcde", 2, 3, "abcdefghij", 0, 10, 2);
    test("abcde", 2, 3, "abcdefghij", 0, 11, 2);
    test("abcde", 2, 3, "abcdefghij", 1, 0, 3);
    test("abcde", 2, 3, "abcdefghij", 1, 1, 1);
    test("abcde", 2, 3, "abcdefghij", 1, 4, 1);
    test("abcde", 2, 3, "abcdefghij", 1, 8, 1);
    test("abcde", 2, 3, "abcdefghij", 1, 9, 1);
    test("abcde", 2, 3, "abcdefghij", 1, 10, 1);
    test("abcde", 2, 3, "abcdefghij", 5, 0, 3);
    test("abcde", 2, 3, "abcdefghij", 5, 1, -3);
    test("abcde", 2, 3, "abcdefghij", 5, 2, -3);
    test("abcde", 2, 3, "abcdefghij", 5, 4, -3);
    test("abcde", 2, 3, "abcdefghij", 5, 5, -3);
    test("abcde", 2, 3, "abcdefghij", 5, 6, -3);
    test("abcde", 2, 3, "abcdefghij", 9, 0, 3);
    test("abcde", 2, 3, "abcdefghij", 9, 1, -7);
    test("abcde", 2, 3, "abcdefghij", 9, 2, -7);
    test("abcde", 2, 3, "abcdefghij", 10, 0, 3);
    test("abcde", 2, 3, "abcdefghij", 10, 1, 3);
    test("abcde", 2, 3, "abcdefghij", 11, 0, 0);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 0, 0, 3);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 0, 1, 2);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 0, 10, 2);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 0, 19, 2);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 0, 20, 2);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 0, 21, 2);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 1, 0, 3);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 1, 1, 1);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 1, 9, 1);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 1, 18, 1);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 1, 19, 1);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 1, 20, 1);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 10, 0, 3);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 10, 1, -8);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 10, 5, -8);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 10, 9, -8);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 10, 10, -8);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 10, 11, -8);
}

void test14()
{
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 19, 0, 3);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 19, 1, -17);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 19, 2, -17);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 20, 0, 3);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 20, 1, 3);
    test("abcde", 2, 3, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 2, 4, "", 0, 0, 3);
    test("abcde", 2, 4, "", 0, 1, 3);
    test("abcde", 2, 4, "", 1, 0, 0);
    test("abcde", 2, 4, "abcde", 0, 0, 3);
    test("abcde", 2, 4, "abcde", 0, 1, 2);
    test("abcde", 2, 4, "abcde", 0, 2, 2);
    test("abcde", 2, 4, "abcde", 0, 4, 2);
    test("abcde", 2, 4, "abcde", 0, 5, 2);
    test("abcde", 2, 4, "abcde", 0, 6, 2);
    test("abcde", 2, 4, "abcde", 1, 0, 3);
    test("abcde", 2, 4, "abcde", 1, 1, 1);
    test("abcde", 2, 4, "abcde", 1, 2, 1);
    test("abcde", 2, 4, "abcde", 1, 3, 1);
    test("abcde", 2, 4, "abcde", 1, 4, 1);
    test("abcde", 2, 4, "abcde", 1, 5, 1);
    test("abcde", 2, 4, "abcde", 2, 0, 3);
    test("abcde", 2, 4, "abcde", 2, 1, 2);
    test("abcde", 2, 4, "abcde", 2, 2, 1);
    test("abcde", 2, 4, "abcde", 2, 3, 0);
    test("abcde", 2, 4, "abcde", 2, 4, 0);
    test("abcde", 2, 4, "abcde", 4, 0, 3);
    test("abcde", 2, 4, "abcde", 4, 1, -2);
    test("abcde", 2, 4, "abcde", 4, 2, -2);
    test("abcde", 2, 4, "abcde", 5, 0, 3);
    test("abcde", 2, 4, "abcde", 5, 1, 3);
    test("abcde", 2, 4, "abcde", 6, 0, 0);
    test("abcde", 2, 4, "abcdefghij", 0, 0, 3);
    test("abcde", 2, 4, "abcdefghij", 0, 1, 2);
    test("abcde", 2, 4, "abcdefghij", 0, 5, 2);
    test("abcde", 2, 4, "abcdefghij", 0, 9, 2);
    test("abcde", 2, 4, "abcdefghij", 0, 10, 2);
    test("abcde", 2, 4, "abcdefghij", 0, 11, 2);
    test("abcde", 2, 4, "abcdefghij", 1, 0, 3);
    test("abcde", 2, 4, "abcdefghij", 1, 1, 1);
    test("abcde", 2, 4, "abcdefghij", 1, 4, 1);
    test("abcde", 2, 4, "abcdefghij", 1, 8, 1);
    test("abcde", 2, 4, "abcdefghij", 1, 9, 1);
    test("abcde", 2, 4, "abcdefghij", 1, 10, 1);
    test("abcde", 2, 4, "abcdefghij", 5, 0, 3);
    test("abcde", 2, 4, "abcdefghij", 5, 1, -3);
    test("abcde", 2, 4, "abcdefghij", 5, 2, -3);
    test("abcde", 2, 4, "abcdefghij", 5, 4, -3);
    test("abcde", 2, 4, "abcdefghij", 5, 5, -3);
    test("abcde", 2, 4, "abcdefghij", 5, 6, -3);
    test("abcde", 2, 4, "abcdefghij", 9, 0, 3);
    test("abcde", 2, 4, "abcdefghij", 9, 1, -7);
    test("abcde", 2, 4, "abcdefghij", 9, 2, -7);
    test("abcde", 2, 4, "abcdefghij", 10, 0, 3);
    test("abcde", 2, 4, "abcdefghij", 10, 1, 3);
    test("abcde", 2, 4, "abcdefghij", 11, 0, 0);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 0, 0, 3);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 0, 1, 2);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 0, 10, 2);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 0, 19, 2);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 0, 20, 2);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 0, 21, 2);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 1, 0, 3);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 1, 1, 1);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 1, 9, 1);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 1, 18, 1);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 1, 19, 1);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 1, 20, 1);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 10, 0, 3);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 10, 1, -8);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 10, 5, -8);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 10, 9, -8);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 10, 10, -8);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 10, 11, -8);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 19, 0, 3);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 19, 1, -17);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 19, 2, -17);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 20, 0, 3);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 20, 1, 3);
    test("abcde", 2, 4, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 4, 0, "", 0, 0, 0);
    test("abcde", 4, 0, "", 0, 1, 0);
    test("abcde", 4, 0, "", 1, 0, 0);
    test("abcde", 4, 0, "abcde", 0, 0, 0);
    test("abcde", 4, 0, "abcde", 0, 1, -1);
    test("abcde", 4, 0, "abcde", 0, 2, -2);
    test("abcde", 4, 0, "abcde", 0, 4, -4);
    test("abcde", 4, 0, "abcde", 0, 5, -5);
    test("abcde", 4, 0, "abcde", 0, 6, -5);
    test("abcde", 4, 0, "abcde", 1, 0, 0);
    test("abcde", 4, 0, "abcde", 1, 1, -1);
    test("abcde", 4, 0, "abcde", 1, 2, -2);
    test("abcde", 4, 0, "abcde", 1, 3, -3);
    test("abcde", 4, 0, "abcde", 1, 4, -4);
    test("abcde", 4, 0, "abcde", 1, 5, -4);
    test("abcde", 4, 0, "abcde", 2, 0, 0);
    test("abcde", 4, 0, "abcde", 2, 1, -1);
    test("abcde", 4, 0, "abcde", 2, 2, -2);
    test("abcde", 4, 0, "abcde", 2, 3, -3);
    test("abcde", 4, 0, "abcde", 2, 4, -3);
}

void test15()
{
    test("abcde", 4, 0, "abcde", 4, 0, 0);
    test("abcde", 4, 0, "abcde", 4, 1, -1);
    test("abcde", 4, 0, "abcde", 4, 2, -1);
    test("abcde", 4, 0, "abcde", 5, 0, 0);
    test("abcde", 4, 0, "abcde", 5, 1, 0);
    test("abcde", 4, 0, "abcde", 6, 0, 0);
    test("abcde", 4, 0, "abcdefghij", 0, 0, 0);
    test("abcde", 4, 0, "abcdefghij", 0, 1, -1);
    test("abcde", 4, 0, "abcdefghij", 0, 5, -5);
    test("abcde", 4, 0, "abcdefghij", 0, 9, -9);
    test("abcde", 4, 0, "abcdefghij", 0, 10, -10);
    test("abcde", 4, 0, "abcdefghij", 0, 11, -10);
    test("abcde", 4, 0, "abcdefghij", 1, 0, 0);
    test("abcde", 4, 0, "abcdefghij", 1, 1, -1);
    test("abcde", 4, 0, "abcdefghij", 1, 4, -4);
    test("abcde", 4, 0, "abcdefghij", 1, 8, -8);
    test("abcde", 4, 0, "abcdefghij", 1, 9, -9);
    test("abcde", 4, 0, "abcdefghij", 1, 10, -9);
    test("abcde", 4, 0, "abcdefghij", 5, 0, 0);
    test("abcde", 4, 0, "abcdefghij", 5, 1, -1);
    test("abcde", 4, 0, "abcdefghij", 5, 2, -2);
    test("abcde", 4, 0, "abcdefghij", 5, 4, -4);
    test("abcde", 4, 0, "abcdefghij", 5, 5, -5);
    test("abcde", 4, 0, "abcdefghij", 5, 6, -5);
    test("abcde", 4, 0, "abcdefghij", 9, 0, 0);
    test("abcde", 4, 0, "abcdefghij", 9, 1, -1);
    test("abcde", 4, 0, "abcdefghij", 9, 2, -1);
    test("abcde", 4, 0, "abcdefghij", 10, 0, 0);
    test("abcde", 4, 0, "abcdefghij", 10, 1, 0);
    test("abcde", 4, 0, "abcdefghij", 11, 0, 0);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 0, 0, 0);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 0, 1, -1);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 0, 10, -10);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 0, 19, -19);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 0, 20, -20);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 0, 21, -20);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 1, 0, 0);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 1, 9, -9);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 1, 18, -18);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 1, 19, -19);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 1, 20, -19);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 10, 0, 0);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 10, 1, -1);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 19, 0, 0);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 19, 1, -1);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 19, 2, -1);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 20, 0, 0);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 20, 1, 0);
    test("abcde", 4, 0, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 4, 1, "", 0, 0, 1);
    test("abcde", 4, 1, "", 0, 1, 1);
    test("abcde", 4, 1, "", 1, 0, 0);
    test("abcde", 4, 1, "abcde", 0, 0, 1);
    test("abcde", 4, 1, "abcde", 0, 1, 4);
    test("abcde", 4, 1, "abcde", 0, 2, 4);
    test("abcde", 4, 1, "abcde", 0, 4, 4);
    test("abcde", 4, 1, "abcde", 0, 5, 4);
    test("abcde", 4, 1, "abcde", 0, 6, 4);
    test("abcde", 4, 1, "abcde", 1, 0, 1);
    test("abcde", 4, 1, "abcde", 1, 1, 3);
    test("abcde", 4, 1, "abcde", 1, 2, 3);
    test("abcde", 4, 1, "abcde", 1, 3, 3);
    test("abcde", 4, 1, "abcde", 1, 4, 3);
    test("abcde", 4, 1, "abcde", 1, 5, 3);
    test("abcde", 4, 1, "abcde", 2, 0, 1);
    test("abcde", 4, 1, "abcde", 2, 1, 2);
    test("abcde", 4, 1, "abcde", 2, 2, 2);
    test("abcde", 4, 1, "abcde", 2, 3, 2);
    test("abcde", 4, 1, "abcde", 2, 4, 2);
    test("abcde", 4, 1, "abcde", 4, 0, 1);
    test("abcde", 4, 1, "abcde", 4, 1, 0);
    test("abcde", 4, 1, "abcde", 4, 2, 0);
    test("abcde", 4, 1, "abcde", 5, 0, 1);
    test("abcde", 4, 1, "abcde", 5, 1, 1);
    test("abcde", 4, 1, "abcde", 6, 0, 0);
    test("abcde", 4, 1, "abcdefghij", 0, 0, 1);
    test("abcde", 4, 1, "abcdefghij", 0, 1, 4);
    test("abcde", 4, 1, "abcdefghij", 0, 5, 4);
    test("abcde", 4, 1, "abcdefghij", 0, 9, 4);
    test("abcde", 4, 1, "abcdefghij", 0, 10, 4);
    test("abcde", 4, 1, "abcdefghij", 0, 11, 4);
    test("abcde", 4, 1, "abcdefghij", 1, 0, 1);
    test("abcde", 4, 1, "abcdefghij", 1, 1, 3);
    test("abcde", 4, 1, "abcdefghij", 1, 4, 3);
    test("abcde", 4, 1, "abcdefghij", 1, 8, 3);
    test("abcde", 4, 1, "abcdefghij", 1, 9, 3);
    test("abcde", 4, 1, "abcdefghij", 1, 10, 3);
    test("abcde", 4, 1, "abcdefghij", 5, 0, 1);
    test("abcde", 4, 1, "abcdefghij", 5, 1, -1);
    test("abcde", 4, 1, "abcdefghij", 5, 2, -1);
    test("abcde", 4, 1, "abcdefghij", 5, 4, -1);
    test("abcde", 4, 1, "abcdefghij", 5, 5, -1);
    test("abcde", 4, 1, "abcdefghij", 5, 6, -1);
    test("abcde", 4, 1, "abcdefghij", 9, 0, 1);
    test("abcde", 4, 1, "abcdefghij", 9, 1, -5);
}

void test16()
{
    test("abcde", 4, 1, "abcdefghij", 9, 2, -5);
    test("abcde", 4, 1, "abcdefghij", 10, 0, 1);
    test("abcde", 4, 1, "abcdefghij", 10, 1, 1);
    test("abcde", 4, 1, "abcdefghij", 11, 0, 0);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 0, 0, 1);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 0, 1, 4);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 0, 10, 4);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 0, 19, 4);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 0, 20, 4);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 0, 21, 4);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 1, 0, 1);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 1, 1, 3);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 1, 9, 3);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 1, 18, 3);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 1, 19, 3);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 1, 20, 3);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 10, 0, 1);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 10, 1, -6);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 10, 5, -6);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 10, 9, -6);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 10, 10, -6);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 10, 11, -6);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 19, 0, 1);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 19, 1, -15);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 19, 2, -15);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 20, 0, 1);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 20, 1, 1);
    test("abcde", 4, 1, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 4, 2, "", 0, 0, 1);
    test("abcde", 4, 2, "", 0, 1, 1);
    test("abcde", 4, 2, "", 1, 0, 0);
    test("abcde", 4, 2, "abcde", 0, 0, 1);
    test("abcde", 4, 2, "abcde", 0, 1, 4);
    test("abcde", 4, 2, "abcde", 0, 2, 4);
    test("abcde", 4, 2, "abcde", 0, 4, 4);
    test("abcde", 4, 2, "abcde", 0, 5, 4);
    test("abcde", 4, 2, "abcde", 0, 6, 4);
    test("abcde", 4, 2, "abcde", 1, 0, 1);
    test("abcde", 4, 2, "abcde", 1, 1, 3);
    test("abcde", 4, 2, "abcde", 1, 2, 3);
    test("abcde", 4, 2, "abcde", 1, 3, 3);
    test("abcde", 4, 2, "abcde", 1, 4, 3);
    test("abcde", 4, 2, "abcde", 1, 5, 3);
    test("abcde", 4, 2, "abcde", 2, 0, 1);
    test("abcde", 4, 2, "abcde", 2, 1, 2);
    test("abcde", 4, 2, "abcde", 2, 2, 2);
    test("abcde", 4, 2, "abcde", 2, 3, 2);
    test("abcde", 4, 2, "abcde", 2, 4, 2);
    test("abcde", 4, 2, "abcde", 4, 0, 1);
    test("abcde", 4, 2, "abcde", 4, 1, 0);
    test("abcde", 4, 2, "abcde", 4, 2, 0);
    test("abcde", 4, 2, "abcde", 5, 0, 1);
    test("abcde", 4, 2, "abcde", 5, 1, 1);
    test("abcde", 4, 2, "abcde", 6, 0, 0);
    test("abcde", 4, 2, "abcdefghij", 0, 0, 1);
    test("abcde", 4, 2, "abcdefghij", 0, 1, 4);
    test("abcde", 4, 2, "abcdefghij", 0, 5, 4);
    test("abcde", 4, 2, "abcdefghij", 0, 9, 4);
    test("abcde", 4, 2, "abcdefghij", 0, 10, 4);
    test("abcde", 4, 2, "abcdefghij", 0, 11, 4);
    test("abcde", 4, 2, "abcdefghij", 1, 0, 1);
    test("abcde", 4, 2, "abcdefghij", 1, 1, 3);
    test("abcde", 4, 2, "abcdefghij", 1, 4, 3);
    test("abcde", 4, 2, "abcdefghij", 1, 8, 3);
    test("abcde", 4, 2, "abcdefghij", 1, 9, 3);
    test("abcde", 4, 2, "abcdefghij", 1, 10, 3);
    test("abcde", 4, 2, "abcdefghij", 5, 0, 1);
    test("abcde", 4, 2, "abcdefghij", 5, 1, -1);
    test("abcde", 4, 2, "abcdefghij", 5, 2, -1);
    test("abcde", 4, 2, "abcdefghij", 5, 4, -1);
    test("abcde", 4, 2, "abcdefghij", 5, 5, -1);
    test("abcde", 4, 2, "abcdefghij", 5, 6, -1);
    test("abcde", 4, 2, "abcdefghij", 9, 0, 1);
    test("abcde", 4, 2, "abcdefghij", 9, 1, -5);
    test("abcde", 4, 2, "abcdefghij", 9, 2, -5);
    test("abcde", 4, 2, "abcdefghij", 10, 0, 1);
    test("abcde", 4, 2, "abcdefghij", 10, 1, 1);
    test("abcde", 4, 2, "abcdefghij", 11, 0, 0);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 0, 0, 1);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 0, 1, 4);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 0, 10, 4);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 0, 19, 4);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 0, 20, 4);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 0, 21, 4);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 1, 0, 1);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 1, 1, 3);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 1, 9, 3);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 1, 18, 3);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 1, 19, 3);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 1, 20, 3);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 10, 0, 1);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 10, 1, -6);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 10, 5, -6);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 10, 9, -6);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 10, 10, -6);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 10, 11, -6);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 19, 0, 1);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 19, 1, -15);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 19, 2, -15);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 20, 0, 1);
}

void test17()
{
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 20, 1, 1);
    test("abcde", 4, 2, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 5, 0, "", 0, 0, 0);
    test("abcde", 5, 0, "", 0, 1, 0);
    test("abcde", 5, 0, "", 1, 0, 0);
    test("abcde", 5, 0, "abcde", 0, 0, 0);
    test("abcde", 5, 0, "abcde", 0, 1, -1);
    test("abcde", 5, 0, "abcde", 0, 2, -2);
    test("abcde", 5, 0, "abcde", 0, 4, -4);
    test("abcde", 5, 0, "abcde", 0, 5, -5);
    test("abcde", 5, 0, "abcde", 0, 6, -5);
    test("abcde", 5, 0, "abcde", 1, 0, 0);
    test("abcde", 5, 0, "abcde", 1, 1, -1);
    test("abcde", 5, 0, "abcde", 1, 2, -2);
    test("abcde", 5, 0, "abcde", 1, 3, -3);
    test("abcde", 5, 0, "abcde", 1, 4, -4);
    test("abcde", 5, 0, "abcde", 1, 5, -4);
    test("abcde", 5, 0, "abcde", 2, 0, 0);
    test("abcde", 5, 0, "abcde", 2, 1, -1);
    test("abcde", 5, 0, "abcde", 2, 2, -2);
    test("abcde", 5, 0, "abcde", 2, 3, -3);
    test("abcde", 5, 0, "abcde", 2, 4, -3);
    test("abcde", 5, 0, "abcde", 4, 0, 0);
    test("abcde", 5, 0, "abcde", 4, 1, -1);
    test("abcde", 5, 0, "abcde", 4, 2, -1);
    test("abcde", 5, 0, "abcde", 5, 0, 0);
    test("abcde", 5, 0, "abcde", 5, 1, 0);
    test("abcde", 5, 0, "abcde", 6, 0, 0);
    test("abcde", 5, 0, "abcdefghij", 0, 0, 0);
    test("abcde", 5, 0, "abcdefghij", 0, 1, -1);
    test("abcde", 5, 0, "abcdefghij", 0, 5, -5);
    test("abcde", 5, 0, "abcdefghij", 0, 9, -9);
    test("abcde", 5, 0, "abcdefghij", 0, 10, -10);
    test("abcde", 5, 0, "abcdefghij", 0, 11, -10);
    test("abcde", 5, 0, "abcdefghij", 1, 0, 0);
    test("abcde", 5, 0, "abcdefghij", 1, 1, -1);
    test("abcde", 5, 0, "abcdefghij", 1, 4, -4);
    test("abcde", 5, 0, "abcdefghij", 1, 8, -8);
    test("abcde", 5, 0, "abcdefghij", 1, 9, -9);
    test("abcde", 5, 0, "abcdefghij", 1, 10, -9);
    test("abcde", 5, 0, "abcdefghij", 5, 0, 0);
    test("abcde", 5, 0, "abcdefghij", 5, 1, -1);
    test("abcde", 5, 0, "abcdefghij", 5, 2, -2);
    test("abcde", 5, 0, "abcdefghij", 5, 4, -4);
    test("abcde", 5, 0, "abcdefghij", 5, 5, -5);
    test("abcde", 5, 0, "abcdefghij", 5, 6, -5);
    test("abcde", 5, 0, "abcdefghij", 9, 0, 0);
    test("abcde", 5, 0, "abcdefghij", 9, 1, -1);
    test("abcde", 5, 0, "abcdefghij", 9, 2, -1);
    test("abcde", 5, 0, "abcdefghij", 10, 0, 0);
    test("abcde", 5, 0, "abcdefghij", 10, 1, 0);
    test("abcde", 5, 0, "abcdefghij", 11, 0, 0);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 0, 0, 0);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 0, 1, -1);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 0, 10, -10);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 0, 19, -19);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 0, 20, -20);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 0, 21, -20);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 1, 0, 0);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 1, 9, -9);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 1, 18, -18);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 1, 19, -19);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 1, 20, -19);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 10, 0, 0);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 10, 1, -1);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 19, 0, 0);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 19, 1, -1);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 19, 2, -1);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 20, 0, 0);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 20, 1, 0);
    test("abcde", 5, 0, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 5, 1, "", 0, 0, 0);
    test("abcde", 5, 1, "", 0, 1, 0);
    test("abcde", 5, 1, "", 1, 0, 0);
    test("abcde", 5, 1, "abcde", 0, 0, 0);
    test("abcde", 5, 1, "abcde", 0, 1, -1);
    test("abcde", 5, 1, "abcde", 0, 2, -2);
    test("abcde", 5, 1, "abcde", 0, 4, -4);
    test("abcde", 5, 1, "abcde", 0, 5, -5);
    test("abcde", 5, 1, "abcde", 0, 6, -5);
    test("abcde", 5, 1, "abcde", 1, 0, 0);
    test("abcde", 5, 1, "abcde", 1, 1, -1);
    test("abcde", 5, 1, "abcde", 1, 2, -2);
    test("abcde", 5, 1, "abcde", 1, 3, -3);
    test("abcde", 5, 1, "abcde", 1, 4, -4);
    test("abcde", 5, 1, "abcde", 1, 5, -4);
    test("abcde", 5, 1, "abcde", 2, 0, 0);
    test("abcde", 5, 1, "abcde", 2, 1, -1);
    test("abcde", 5, 1, "abcde", 2, 2, -2);
    test("abcde", 5, 1, "abcde", 2, 3, -3);
    test("abcde", 5, 1, "abcde", 2, 4, -3);
    test("abcde", 5, 1, "abcde", 4, 0, 0);
    test("abcde", 5, 1, "abcde", 4, 1, -1);
    test("abcde", 5, 1, "abcde", 4, 2, -1);
    test("abcde", 5, 1, "abcde", 5, 0, 0);
}

void test18()
{
    test("abcde", 5, 1, "abcde", 5, 1, 0);
    test("abcde", 5, 1, "abcde", 6, 0, 0);
    test("abcde", 5, 1, "abcdefghij", 0, 0, 0);
    test("abcde", 5, 1, "abcdefghij", 0, 1, -1);
    test("abcde", 5, 1, "abcdefghij", 0, 5, -5);
    test("abcde", 5, 1, "abcdefghij", 0, 9, -9);
    test("abcde", 5, 1, "abcdefghij", 0, 10, -10);
    test("abcde", 5, 1, "abcdefghij", 0, 11, -10);
    test("abcde", 5, 1, "abcdefghij", 1, 0, 0);
    test("abcde", 5, 1, "abcdefghij", 1, 1, -1);
    test("abcde", 5, 1, "abcdefghij", 1, 4, -4);
    test("abcde", 5, 1, "abcdefghij", 1, 8, -8);
    test("abcde", 5, 1, "abcdefghij", 1, 9, -9);
    test("abcde", 5, 1, "abcdefghij", 1, 10, -9);
    test("abcde", 5, 1, "abcdefghij", 5, 0, 0);
    test("abcde", 5, 1, "abcdefghij", 5, 1, -1);
    test("abcde", 5, 1, "abcdefghij", 5, 2, -2);
    test("abcde", 5, 1, "abcdefghij", 5, 4, -4);
    test("abcde", 5, 1, "abcdefghij", 5, 5, -5);
    test("abcde", 5, 1, "abcdefghij", 5, 6, -5);
    test("abcde", 5, 1, "abcdefghij", 9, 0, 0);
    test("abcde", 5, 1, "abcdefghij", 9, 1, -1);
    test("abcde", 5, 1, "abcdefghij", 9, 2, -1);
    test("abcde", 5, 1, "abcdefghij", 10, 0, 0);
    test("abcde", 5, 1, "abcdefghij", 10, 1, 0);
    test("abcde", 5, 1, "abcdefghij", 11, 0, 0);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 0, 0, 0);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 0, 1, -1);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 0, 10, -10);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 0, 19, -19);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 0, 20, -20);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 0, 21, -20);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 1, 0, 0);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 1, 9, -9);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 1, 18, -18);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 1, 19, -19);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 1, 20, -19);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 10, 0, 0);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 10, 1, -1);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 19, 0, 0);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 19, 1, -1);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 19, 2, -1);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 20, 0, 0);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 20, 1, 0);
    test("abcde", 5, 1, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcde", 6, 0, "", 0, 0, 0);
    test("abcde", 6, 0, "", 0, 1, 0);
    test("abcde", 6, 0, "", 1, 0, 0);
    test("abcde", 6, 0, "abcde", 0, 0, 0);
    test("abcde", 6, 0, "abcde", 0, 1, 0);
    test("abcde", 6, 0, "abcde", 0, 2, 0);
    test("abcde", 6, 0, "abcde", 0, 4, 0);
    test("abcde", 6, 0, "abcde", 0, 5, 0);
    test("abcde", 6, 0, "abcde", 0, 6, 0);
    test("abcde", 6, 0, "abcde", 1, 0, 0);
    test("abcde", 6, 0, "abcde", 1, 1, 0);
    test("abcde", 6, 0, "abcde", 1, 2, 0);
    test("abcde", 6, 0, "abcde", 1, 3, 0);
    test("abcde", 6, 0, "abcde", 1, 4, 0);
    test("abcde", 6, 0, "abcde", 1, 5, 0);
    test("abcde", 6, 0, "abcde", 2, 0, 0);
    test("abcde", 6, 0, "abcde", 2, 1, 0);
    test("abcde", 6, 0, "abcde", 2, 2, 0);
    test("abcde", 6, 0, "abcde", 2, 3, 0);
    test("abcde", 6, 0, "abcde", 2, 4, 0);
    test("abcde", 6, 0, "abcde", 4, 0, 0);
    test("abcde", 6, 0, "abcde", 4, 1, 0);
    test("abcde", 6, 0, "abcde", 4, 2, 0);
    test("abcde", 6, 0, "abcde", 5, 0, 0);
    test("abcde", 6, 0, "abcde", 5, 1, 0);
    test("abcde", 6, 0, "abcde", 6, 0, 0);
    test("abcde", 6, 0, "abcdefghij", 0, 0, 0);
    test("abcde", 6, 0, "abcdefghij", 0, 1, 0);
    test("abcde", 6, 0, "abcdefghij", 0, 5, 0);
    test("abcde", 6, 0, "abcdefghij", 0, 9, 0);
    test("abcde", 6, 0, "abcdefghij", 0, 10, 0);
    test("abcde", 6, 0, "abcdefghij", 0, 11, 0);
    test("abcde", 6, 0, "abcdefghij", 1, 0, 0);
    test("abcde", 6, 0, "abcdefghij", 1, 1, 0);
    test("abcde", 6, 0, "abcdefghij", 1, 4, 0);
    test("abcde", 6, 0, "abcdefghij", 1, 8, 0);
    test("abcde", 6, 0, "abcdefghij", 1, 9, 0);
    test("abcde", 6, 0, "abcdefghij", 1, 10, 0);
    test("abcde", 6, 0, "abcdefghij", 5, 0, 0);
    test("abcde", 6, 0, "abcdefghij", 5, 1, 0);
    test("abcde", 6, 0, "abcdefghij", 5, 2, 0);
    test("abcde", 6, 0, "abcdefghij", 5, 4, 0);
    test("abcde", 6, 0, "abcdefghij", 5, 5, 0);
    test("abcde", 6, 0, "abcdefghij", 5, 6, 0);
    test("abcde", 6, 0, "abcdefghij", 9, 0, 0);
    test("abcde", 6, 0, "abcdefghij", 9, 1, 0);
    test("abcde", 6, 0, "abcdefghij", 9, 2, 0);
    test("abcde", 6, 0, "abcdefghij", 10, 0, 0);
    test("abcde", 6, 0, "abcdefghij", 10, 1, 0);
    test("abcde", 6, 0, "abcdefghij", 11, 0, 0);
}

void test19()
{
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 0, 0, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 0, 1, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 0, 10, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 0, 19, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 0, 20, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 0, 21, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 1, 0, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 1, 1, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 1, 9, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 1, 18, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 1, 19, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 1, 20, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 10, 0, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 10, 1, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 10, 5, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 10, 9, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 10, 10, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 10, 11, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 19, 0, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 19, 1, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 19, 2, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 20, 0, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 20, 1, 0);
    test("abcde", 6, 0, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 0, 0, "", 0, 0, 0);
    test("abcdefghij", 0, 0, "", 0, 1, 0);
    test("abcdefghij", 0, 0, "", 1, 0, 0);
    test("abcdefghij", 0, 0, "abcde", 0, 0, 0);
    test("abcdefghij", 0, 0, "abcde", 0, 1, -1);
    test("abcdefghij", 0, 0, "abcde", 0, 2, -2);
    test("abcdefghij", 0, 0, "abcde", 0, 4, -4);
    test("abcdefghij", 0, 0, "abcde", 0, 5, -5);
    test("abcdefghij", 0, 0, "abcde", 0, 6, -5);
    test("abcdefghij", 0, 0, "abcde", 1, 0, 0);
    test("abcdefghij", 0, 0, "abcde", 1, 1, -1);
    test("abcdefghij", 0, 0, "abcde", 1, 2, -2);
    test("abcdefghij", 0, 0, "abcde", 1, 3, -3);
    test("abcdefghij", 0, 0, "abcde", 1, 4, -4);
    test("abcdefghij", 0, 0, "abcde", 1, 5, -4);
    test("abcdefghij", 0, 0, "abcde", 2, 0, 0);
    test("abcdefghij", 0, 0, "abcde", 2, 1, -1);
    test("abcdefghij", 0, 0, "abcde", 2, 2, -2);
    test("abcdefghij", 0, 0, "abcde", 2, 3, -3);
    test("abcdefghij", 0, 0, "abcde", 2, 4, -3);
    test("abcdefghij", 0, 0, "abcde", 4, 0, 0);
    test("abcdefghij", 0, 0, "abcde", 4, 1, -1);
    test("abcdefghij", 0, 0, "abcde", 4, 2, -1);
    test("abcdefghij", 0, 0, "abcde", 5, 0, 0);
    test("abcdefghij", 0, 0, "abcde", 5, 1, 0);
    test("abcdefghij", 0, 0, "abcde", 6, 0, 0);
    test("abcdefghij", 0, 0, "abcdefghij", 0, 0, 0);
    test("abcdefghij", 0, 0, "abcdefghij", 0, 1, -1);
    test("abcdefghij", 0, 0, "abcdefghij", 0, 5, -5);
    test("abcdefghij", 0, 0, "abcdefghij", 0, 9, -9);
    test("abcdefghij", 0, 0, "abcdefghij", 0, 10, -10);
    test("abcdefghij", 0, 0, "abcdefghij", 0, 11, -10);
    test("abcdefghij", 0, 0, "abcdefghij", 1, 0, 0);
    test("abcdefghij", 0, 0, "abcdefghij", 1, 1, -1);
    test("abcdefghij", 0, 0, "abcdefghij", 1, 4, -4);
    test("abcdefghij", 0, 0, "abcdefghij", 1, 8, -8);
    test("abcdefghij", 0, 0, "abcdefghij", 1, 9, -9);
    test("abcdefghij", 0, 0, "abcdefghij", 1, 10, -9);
    test("abcdefghij", 0, 0, "abcdefghij", 5, 0, 0);
    test("abcdefghij", 0, 0, "abcdefghij", 5, 1, -1);
    test("abcdefghij", 0, 0, "abcdefghij", 5, 2, -2);
    test("abcdefghij", 0, 0, "abcdefghij", 5, 4, -4);
    test("abcdefghij", 0, 0, "abcdefghij", 5, 5, -5);
    test("abcdefghij", 0, 0, "abcdefghij", 5, 6, -5);
    test("abcdefghij", 0, 0, "abcdefghij", 9, 0, 0);
    test("abcdefghij", 0, 0, "abcdefghij", 9, 1, -1);
    test("abcdefghij", 0, 0, "abcdefghij", 9, 2, -1);
    test("abcdefghij", 0, 0, "abcdefghij", 10, 0, 0);
    test("abcdefghij", 0, 0, "abcdefghij", 10, 1, 0);
    test("abcdefghij", 0, 0, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 0, 0, 0);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 0, 1, -1);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 0, 10, -10);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 0, 19, -19);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 0, 20, -20);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 0, 21, -20);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 1, 0, 0);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 1, 9, -9);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 1, 18, -18);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 1, 19, -19);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 1, 20, -19);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 10, 0, 0);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 10, 1, -1);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 19, 0, 0);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 19, 1, -1);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 19, 2, -1);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 20, 0, 0);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 20, 1, 0);
    test("abcdefghij", 0, 0, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 0, 1, "", 0, 0, 1);
    test("abcdefghij", 0, 1, "", 0, 1, 1);
}

void test20()
{
    test("abcdefghij", 0, 1, "", 1, 0, 0);
    test("abcdefghij", 0, 1, "abcde", 0, 0, 1);
    test("abcdefghij", 0, 1, "abcde", 0, 1, 0);
    test("abcdefghij", 0, 1, "abcde", 0, 2, -1);
    test("abcdefghij", 0, 1, "abcde", 0, 4, -3);
    test("abcdefghij", 0, 1, "abcde", 0, 5, -4);
    test("abcdefghij", 0, 1, "abcde", 0, 6, -4);
    test("abcdefghij", 0, 1, "abcde", 1, 0, 1);
    test("abcdefghij", 0, 1, "abcde", 1, 1, -1);
    test("abcdefghij", 0, 1, "abcde", 1, 2, -1);
    test("abcdefghij", 0, 1, "abcde", 1, 3, -1);
    test("abcdefghij", 0, 1, "abcde", 1, 4, -1);
    test("abcdefghij", 0, 1, "abcde", 1, 5, -1);
    test("abcdefghij", 0, 1, "abcde", 2, 0, 1);
    test("abcdefghij", 0, 1, "abcde", 2, 1, -2);
    test("abcdefghij", 0, 1, "abcde", 2, 2, -2);
    test("abcdefghij", 0, 1, "abcde", 2, 3, -2);
    test("abcdefghij", 0, 1, "abcde", 2, 4, -2);
    test("abcdefghij", 0, 1, "abcde", 4, 0, 1);
    test("abcdefghij", 0, 1, "abcde", 4, 1, -4);
    test("abcdefghij", 0, 1, "abcde", 4, 2, -4);
    test("abcdefghij", 0, 1, "abcde", 5, 0, 1);
    test("abcdefghij", 0, 1, "abcde", 5, 1, 1);
    test("abcdefghij", 0, 1, "abcde", 6, 0, 0);
    test("abcdefghij", 0, 1, "abcdefghij", 0, 0, 1);
    test("abcdefghij", 0, 1, "abcdefghij", 0, 1, 0);
    test("abcdefghij", 0, 1, "abcdefghij", 0, 5, -4);
    test("abcdefghij", 0, 1, "abcdefghij", 0, 9, -8);
    test("abcdefghij", 0, 1, "abcdefghij", 0, 10, -9);
    test("abcdefghij", 0, 1, "abcdefghij", 0, 11, -9);
    test("abcdefghij", 0, 1, "abcdefghij", 1, 0, 1);
    test("abcdefghij", 0, 1, "abcdefghij", 1, 1, -1);
    test("abcdefghij", 0, 1, "abcdefghij", 1, 4, -1);
    test("abcdefghij", 0, 1, "abcdefghij", 1, 8, -1);
    test("abcdefghij", 0, 1, "abcdefghij", 1, 9, -1);
    test("abcdefghij", 0, 1, "abcdefghij", 1, 10, -1);
    test("abcdefghij", 0, 1, "abcdefghij", 5, 0, 1);
    test("abcdefghij", 0, 1, "abcdefghij", 5, 1, -5);
    test("abcdefghij", 0, 1, "abcdefghij", 5, 2, -5);
    test("abcdefghij", 0, 1, "abcdefghij", 5, 4, -5);
    test("abcdefghij", 0, 1, "abcdefghij", 5, 5, -5);
    test("abcdefghij", 0, 1, "abcdefghij", 5, 6, -5);
    test("abcdefghij", 0, 1, "abcdefghij", 9, 0, 1);
    test("abcdefghij", 0, 1, "abcdefghij", 9, 1, -9);
    test("abcdefghij", 0, 1, "abcdefghij", 9, 2, -9);
    test("abcdefghij", 0, 1, "abcdefghij", 10, 0, 1);
    test("abcdefghij", 0, 1, "abcdefghij", 10, 1, 1);
    test("abcdefghij", 0, 1, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 0, 0, 1);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 0, 1, 0);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 0, 10, -9);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 0, 19, -18);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 0, 20, -19);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 0, 21, -19);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 1, 0, 1);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 1, 9, -1);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 1, 18, -1);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 1, 19, -1);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 1, 20, -1);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 10, 0, 1);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 10, 1, -10);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 10, 5, -10);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 10, 9, -10);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 19, 0, 1);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 19, 1, -19);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 19, 2, -19);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 20, 0, 1);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 20, 1, 1);
    test("abcdefghij", 0, 1, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 0, 5, "", 0, 0, 5);
    test("abcdefghij", 0, 5, "", 0, 1, 5);
    test("abcdefghij", 0, 5, "", 1, 0, 0);
    test("abcdefghij", 0, 5, "abcde", 0, 0, 5);
    test("abcdefghij", 0, 5, "abcde", 0, 1, 4);
    test("abcdefghij", 0, 5, "abcde", 0, 2, 3);
    test("abcdefghij", 0, 5, "abcde", 0, 4, 1);
    test("abcdefghij", 0, 5, "abcde", 0, 5, 0);
    test("abcdefghij", 0, 5, "abcde", 0, 6, 0);
    test("abcdefghij", 0, 5, "abcde", 1, 0, 5);
    test("abcdefghij", 0, 5, "abcde", 1, 1, -1);
    test("abcdefghij", 0, 5, "abcde", 1, 2, -1);
    test("abcdefghij", 0, 5, "abcde", 1, 3, -1);
    test("abcdefghij", 0, 5, "abcde", 1, 4, -1);
    test("abcdefghij", 0, 5, "abcde", 1, 5, -1);
    test("abcdefghij", 0, 5, "abcde", 2, 0, 5);
    test("abcdefghij", 0, 5, "abcde", 2, 1, -2);
    test("abcdefghij", 0, 5, "abcde", 2, 2, -2);
    test("abcdefghij", 0, 5, "abcde", 2, 3, -2);
    test("abcdefghij", 0, 5, "abcde", 2, 4, -2);
    test("abcdefghij", 0, 5, "abcde", 4, 0, 5);
    test("abcdefghij", 0, 5, "abcde", 4, 1, -4);
    test("abcdefghij", 0, 5, "abcde", 4, 2, -4);
    test("abcdefghij", 0, 5, "abcde", 5, 0, 5);
    test("abcdefghij", 0, 5, "abcde", 5, 1, 5);
    test("abcdefghij", 0, 5, "abcde", 6, 0, 0);
    test("abcdefghij", 0, 5, "abcdefghij", 0, 0, 5);
    test("abcdefghij", 0, 5, "abcdefghij", 0, 1, 4);
}

void test21()
{
    test("abcdefghij", 0, 5, "abcdefghij", 0, 5, 0);
    test("abcdefghij", 0, 5, "abcdefghij", 0, 9, -4);
    test("abcdefghij", 0, 5, "abcdefghij", 0, 10, -5);
    test("abcdefghij", 0, 5, "abcdefghij", 0, 11, -5);
    test("abcdefghij", 0, 5, "abcdefghij", 1, 0, 5);
    test("abcdefghij", 0, 5, "abcdefghij", 1, 1, -1);
    test("abcdefghij", 0, 5, "abcdefghij", 1, 4, -1);
    test("abcdefghij", 0, 5, "abcdefghij", 1, 8, -1);
    test("abcdefghij", 0, 5, "abcdefghij", 1, 9, -1);
    test("abcdefghij", 0, 5, "abcdefghij", 1, 10, -1);
    test("abcdefghij", 0, 5, "abcdefghij", 5, 0, 5);
    test("abcdefghij", 0, 5, "abcdefghij", 5, 1, -5);
    test("abcdefghij", 0, 5, "abcdefghij", 5, 2, -5);
    test("abcdefghij", 0, 5, "abcdefghij", 5, 4, -5);
    test("abcdefghij", 0, 5, "abcdefghij", 5, 5, -5);
    test("abcdefghij", 0, 5, "abcdefghij", 5, 6, -5);
    test("abcdefghij", 0, 5, "abcdefghij", 9, 0, 5);
    test("abcdefghij", 0, 5, "abcdefghij", 9, 1, -9);
    test("abcdefghij", 0, 5, "abcdefghij", 9, 2, -9);
    test("abcdefghij", 0, 5, "abcdefghij", 10, 0, 5);
    test("abcdefghij", 0, 5, "abcdefghij", 10, 1, 5);
    test("abcdefghij", 0, 5, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 0, 0, 5);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 0, 1, 4);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 0, 10, -5);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 0, 19, -14);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 0, 20, -15);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 0, 21, -15);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 1, 0, 5);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 1, 9, -1);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 1, 18, -1);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 1, 19, -1);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 1, 20, -1);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 10, 0, 5);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 10, 1, -10);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 10, 5, -10);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 10, 9, -10);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 19, 0, 5);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 19, 1, -19);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 19, 2, -19);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 20, 0, 5);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 20, 1, 5);
    test("abcdefghij", 0, 5, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 0, 9, "", 0, 0, 9);
    test("abcdefghij", 0, 9, "", 0, 1, 9);
    test("abcdefghij", 0, 9, "", 1, 0, 0);
    test("abcdefghij", 0, 9, "abcde", 0, 0, 9);
    test("abcdefghij", 0, 9, "abcde", 0, 1, 8);
    test("abcdefghij", 0, 9, "abcde", 0, 2, 7);
    test("abcdefghij", 0, 9, "abcde", 0, 4, 5);
    test("abcdefghij", 0, 9, "abcde", 0, 5, 4);
    test("abcdefghij", 0, 9, "abcde", 0, 6, 4);
    test("abcdefghij", 0, 9, "abcde", 1, 0, 9);
    test("abcdefghij", 0, 9, "abcde", 1, 1, -1);
    test("abcdefghij", 0, 9, "abcde", 1, 2, -1);
    test("abcdefghij", 0, 9, "abcde", 1, 3, -1);
    test("abcdefghij", 0, 9, "abcde", 1, 4, -1);
    test("abcdefghij", 0, 9, "abcde", 1, 5, -1);
    test("abcdefghij", 0, 9, "abcde", 2, 0, 9);
    test("abcdefghij", 0, 9, "abcde", 2, 1, -2);
    test("abcdefghij", 0, 9, "abcde", 2, 2, -2);
    test("abcdefghij", 0, 9, "abcde", 2, 3, -2);
    test("abcdefghij", 0, 9, "abcde", 2, 4, -2);
    test("abcdefghij", 0, 9, "abcde", 4, 0, 9);
    test("abcdefghij", 0, 9, "abcde", 4, 1, -4);
    test("abcdefghij", 0, 9, "abcde", 4, 2, -4);
    test("abcdefghij", 0, 9, "abcde", 5, 0, 9);
    test("abcdefghij", 0, 9, "abcde", 5, 1, 9);
    test("abcdefghij", 0, 9, "abcde", 6, 0, 0);
    test("abcdefghij", 0, 9, "abcdefghij", 0, 0, 9);
    test("abcdefghij", 0, 9, "abcdefghij", 0, 1, 8);
    test("abcdefghij", 0, 9, "abcdefghij", 0, 5, 4);
    test("abcdefghij", 0, 9, "abcdefghij", 0, 9, 0);
    test("abcdefghij", 0, 9, "abcdefghij", 0, 10, -1);
    test("abcdefghij", 0, 9, "abcdefghij", 0, 11, -1);
    test("abcdefghij", 0, 9, "abcdefghij", 1, 0, 9);
    test("abcdefghij", 0, 9, "abcdefghij", 1, 1, -1);
    test("abcdefghij", 0, 9, "abcdefghij", 1, 4, -1);
    test("abcdefghij", 0, 9, "abcdefghij", 1, 8, -1);
    test("abcdefghij", 0, 9, "abcdefghij", 1, 9, -1);
    test("abcdefghij", 0, 9, "abcdefghij", 1, 10, -1);
    test("abcdefghij", 0, 9, "abcdefghij", 5, 0, 9);
    test("abcdefghij", 0, 9, "abcdefghij", 5, 1, -5);
    test("abcdefghij", 0, 9, "abcdefghij", 5, 2, -5);
    test("abcdefghij", 0, 9, "abcdefghij", 5, 4, -5);
    test("abcdefghij", 0, 9, "abcdefghij", 5, 5, -5);
    test("abcdefghij", 0, 9, "abcdefghij", 5, 6, -5);
    test("abcdefghij", 0, 9, "abcdefghij", 9, 0, 9);
    test("abcdefghij", 0, 9, "abcdefghij", 9, 1, -9);
    test("abcdefghij", 0, 9, "abcdefghij", 9, 2, -9);
    test("abcdefghij", 0, 9, "abcdefghij", 10, 0, 9);
    test("abcdefghij", 0, 9, "abcdefghij", 10, 1, 9);
    test("abcdefghij", 0, 9, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 0, 0, 9);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 0, 1, 8);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 0, 10, -1);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 0, 19, -10);
}

void test22()
{
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 0, 20, -11);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 0, 21, -11);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 1, 0, 9);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 1, 9, -1);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 1, 18, -1);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 1, 19, -1);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 1, 20, -1);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 10, 0, 9);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 10, 1, -10);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 10, 5, -10);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 10, 9, -10);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 19, 0, 9);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 19, 1, -19);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 19, 2, -19);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 20, 0, 9);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 20, 1, 9);
    test("abcdefghij", 0, 9, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 0, 10, "", 0, 0, 10);
    test("abcdefghij", 0, 10, "", 0, 1, 10);
    test("abcdefghij", 0, 10, "", 1, 0, 0);
    test("abcdefghij", 0, 10, "abcde", 0, 0, 10);
    test("abcdefghij", 0, 10, "abcde", 0, 1, 9);
    test("abcdefghij", 0, 10, "abcde", 0, 2, 8);
    test("abcdefghij", 0, 10, "abcde", 0, 4, 6);
    test("abcdefghij", 0, 10, "abcde", 0, 5, 5);
    test("abcdefghij", 0, 10, "abcde", 0, 6, 5);
    test("abcdefghij", 0, 10, "abcde", 1, 0, 10);
    test("abcdefghij", 0, 10, "abcde", 1, 1, -1);
    test("abcdefghij", 0, 10, "abcde", 1, 2, -1);
    test("abcdefghij", 0, 10, "abcde", 1, 3, -1);
    test("abcdefghij", 0, 10, "abcde", 1, 4, -1);
    test("abcdefghij", 0, 10, "abcde", 1, 5, -1);
    test("abcdefghij", 0, 10, "abcde", 2, 0, 10);
    test("abcdefghij", 0, 10, "abcde", 2, 1, -2);
    test("abcdefghij", 0, 10, "abcde", 2, 2, -2);
    test("abcdefghij", 0, 10, "abcde", 2, 3, -2);
    test("abcdefghij", 0, 10, "abcde", 2, 4, -2);
    test("abcdefghij", 0, 10, "abcde", 4, 0, 10);
    test("abcdefghij", 0, 10, "abcde", 4, 1, -4);
    test("abcdefghij", 0, 10, "abcde", 4, 2, -4);
    test("abcdefghij", 0, 10, "abcde", 5, 0, 10);
    test("abcdefghij", 0, 10, "abcde", 5, 1, 10);
    test("abcdefghij", 0, 10, "abcde", 6, 0, 0);
    test("abcdefghij", 0, 10, "abcdefghij", 0, 0, 10);
    test("abcdefghij", 0, 10, "abcdefghij", 0, 1, 9);
    test("abcdefghij", 0, 10, "abcdefghij", 0, 5, 5);
    test("abcdefghij", 0, 10, "abcdefghij", 0, 9, 1);
    test("abcdefghij", 0, 10, "abcdefghij", 0, 10, 0);
    test("abcdefghij", 0, 10, "abcdefghij", 0, 11, 0);
    test("abcdefghij", 0, 10, "abcdefghij", 1, 0, 10);
    test("abcdefghij", 0, 10, "abcdefghij", 1, 1, -1);
    test("abcdefghij", 0, 10, "abcdefghij", 1, 4, -1);
    test("abcdefghij", 0, 10, "abcdefghij", 1, 8, -1);
    test("abcdefghij", 0, 10, "abcdefghij", 1, 9, -1);
    test("abcdefghij", 0, 10, "abcdefghij", 1, 10, -1);
    test("abcdefghij", 0, 10, "abcdefghij", 5, 0, 10);
    test("abcdefghij", 0, 10, "abcdefghij", 5, 1, -5);
    test("abcdefghij", 0, 10, "abcdefghij", 5, 2, -5);
    test("abcdefghij", 0, 10, "abcdefghij", 5, 4, -5);
    test("abcdefghij", 0, 10, "abcdefghij", 5, 5, -5);
    test("abcdefghij", 0, 10, "abcdefghij", 5, 6, -5);
    test("abcdefghij", 0, 10, "abcdefghij", 9, 0, 10);
    test("abcdefghij", 0, 10, "abcdefghij", 9, 1, -9);
    test("abcdefghij", 0, 10, "abcdefghij", 9, 2, -9);
    test("abcdefghij", 0, 10, "abcdefghij", 10, 0, 10);
    test("abcdefghij", 0, 10, "abcdefghij", 10, 1, 10);
    test("abcdefghij", 0, 10, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 0, 0, 10);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 0, 1, 9);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 0, 10, 0);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 0, 19, -9);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 0, 20, -10);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 0, 21, -10);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 1, 0, 10);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 1, 9, -1);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 1, 18, -1);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 1, 19, -1);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 1, 20, -1);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 10, 0, 10);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 10, 1, -10);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 10, 5, -10);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 10, 9, -10);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 19, 0, 10);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 19, 1, -19);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 19, 2, -19);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 20, 0, 10);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 20, 1, 10);
    test("abcdefghij", 0, 10, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 0, 11, "", 0, 0, 10);
    test("abcdefghij", 0, 11, "", 0, 1, 10);
    test("abcdefghij", 0, 11, "", 1, 0, 0);
    test("abcdefghij", 0, 11, "abcde", 0, 0, 10);
    test("abcdefghij", 0, 11, "abcde", 0, 1, 9);
    test("abcdefghij", 0, 11, "abcde", 0, 2, 8);
}

void test23()
{
    test("abcdefghij", 0, 11, "abcde", 0, 4, 6);
    test("abcdefghij", 0, 11, "abcde", 0, 5, 5);
    test("abcdefghij", 0, 11, "abcde", 0, 6, 5);
    test("abcdefghij", 0, 11, "abcde", 1, 0, 10);
    test("abcdefghij", 0, 11, "abcde", 1, 1, -1);
    test("abcdefghij", 0, 11, "abcde", 1, 2, -1);
    test("abcdefghij", 0, 11, "abcde", 1, 3, -1);
    test("abcdefghij", 0, 11, "abcde", 1, 4, -1);
    test("abcdefghij", 0, 11, "abcde", 1, 5, -1);
    test("abcdefghij", 0, 11, "abcde", 2, 0, 10);
    test("abcdefghij", 0, 11, "abcde", 2, 1, -2);
    test("abcdefghij", 0, 11, "abcde", 2, 2, -2);
    test("abcdefghij", 0, 11, "abcde", 2, 3, -2);
    test("abcdefghij", 0, 11, "abcde", 2, 4, -2);
    test("abcdefghij", 0, 11, "abcde", 4, 0, 10);
    test("abcdefghij", 0, 11, "abcde", 4, 1, -4);
    test("abcdefghij", 0, 11, "abcde", 4, 2, -4);
    test("abcdefghij", 0, 11, "abcde", 5, 0, 10);
    test("abcdefghij", 0, 11, "abcde", 5, 1, 10);
    test("abcdefghij", 0, 11, "abcde", 6, 0, 0);
    test("abcdefghij", 0, 11, "abcdefghij", 0, 0, 10);
    test("abcdefghij", 0, 11, "abcdefghij", 0, 1, 9);
    test("abcdefghij", 0, 11, "abcdefghij", 0, 5, 5);
    test("abcdefghij", 0, 11, "abcdefghij", 0, 9, 1);
    test("abcdefghij", 0, 11, "abcdefghij", 0, 10, 0);
    test("abcdefghij", 0, 11, "abcdefghij", 0, 11, 0);
    test("abcdefghij", 0, 11, "abcdefghij", 1, 0, 10);
    test("abcdefghij", 0, 11, "abcdefghij", 1, 1, -1);
    test("abcdefghij", 0, 11, "abcdefghij", 1, 4, -1);
    test("abcdefghij", 0, 11, "abcdefghij", 1, 8, -1);
    test("abcdefghij", 0, 11, "abcdefghij", 1, 9, -1);
    test("abcdefghij", 0, 11, "abcdefghij", 1, 10, -1);
    test("abcdefghij", 0, 11, "abcdefghij", 5, 0, 10);
    test("abcdefghij", 0, 11, "abcdefghij", 5, 1, -5);
    test("abcdefghij", 0, 11, "abcdefghij", 5, 2, -5);
    test("abcdefghij", 0, 11, "abcdefghij", 5, 4, -5);
    test("abcdefghij", 0, 11, "abcdefghij", 5, 5, -5);
    test("abcdefghij", 0, 11, "abcdefghij", 5, 6, -5);
    test("abcdefghij", 0, 11, "abcdefghij", 9, 0, 10);
    test("abcdefghij", 0, 11, "abcdefghij", 9, 1, -9);
    test("abcdefghij", 0, 11, "abcdefghij", 9, 2, -9);
    test("abcdefghij", 0, 11, "abcdefghij", 10, 0, 10);
    test("abcdefghij", 0, 11, "abcdefghij", 10, 1, 10);
    test("abcdefghij", 0, 11, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 0, 0, 10);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 0, 1, 9);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 0, 10, 0);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 0, 19, -9);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 0, 20, -10);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 0, 21, -10);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 1, 0, 10);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 1, 9, -1);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 1, 18, -1);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 1, 19, -1);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 1, 20, -1);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 10, 0, 10);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 10, 1, -10);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 10, 5, -10);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 10, 9, -10);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 19, 0, 10);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 19, 1, -19);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 19, 2, -19);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 20, 0, 10);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 20, 1, 10);
    test("abcdefghij", 0, 11, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 1, 0, "", 0, 0, 0);
    test("abcdefghij", 1, 0, "", 0, 1, 0);
    test("abcdefghij", 1, 0, "", 1, 0, 0);
    test("abcdefghij", 1, 0, "abcde", 0, 0, 0);
    test("abcdefghij", 1, 0, "abcde", 0, 1, -1);
    test("abcdefghij", 1, 0, "abcde", 0, 2, -2);
    test("abcdefghij", 1, 0, "abcde", 0, 4, -4);
    test("abcdefghij", 1, 0, "abcde", 0, 5, -5);
    test("abcdefghij", 1, 0, "abcde", 0, 6, -5);
    test("abcdefghij", 1, 0, "abcde", 1, 0, 0);
    test("abcdefghij", 1, 0, "abcde", 1, 1, -1);
    test("abcdefghij", 1, 0, "abcde", 1, 2, -2);
    test("abcdefghij", 1, 0, "abcde", 1, 3, -3);
    test("abcdefghij", 1, 0, "abcde", 1, 4, -4);
    test("abcdefghij", 1, 0, "abcde", 1, 5, -4);
    test("abcdefghij", 1, 0, "abcde", 2, 0, 0);
    test("abcdefghij", 1, 0, "abcde", 2, 1, -1);
    test("abcdefghij", 1, 0, "abcde", 2, 2, -2);
    test("abcdefghij", 1, 0, "abcde", 2, 3, -3);
    test("abcdefghij", 1, 0, "abcde", 2, 4, -3);
    test("abcdefghij", 1, 0, "abcde", 4, 0, 0);
    test("abcdefghij", 1, 0, "abcde", 4, 1, -1);
    test("abcdefghij", 1, 0, "abcde", 4, 2, -1);
    test("abcdefghij", 1, 0, "abcde", 5, 0, 0);
    test("abcdefghij", 1, 0, "abcde", 5, 1, 0);
    test("abcdefghij", 1, 0, "abcde", 6, 0, 0);
    test("abcdefghij", 1, 0, "abcdefghij", 0, 0, 0);
    test("abcdefghij", 1, 0, "abcdefghij", 0, 1, -1);
    test("abcdefghij", 1, 0, "abcdefghij", 0, 5, -5);
    test("abcdefghij", 1, 0, "abcdefghij", 0, 9, -9);
    test("abcdefghij", 1, 0, "abcdefghij", 0, 10, -10);
    test("abcdefghij", 1, 0, "abcdefghij", 0, 11, -10);
}

void test24()
{
    test("abcdefghij", 1, 0, "abcdefghij", 1, 0, 0);
    test("abcdefghij", 1, 0, "abcdefghij", 1, 1, -1);
    test("abcdefghij", 1, 0, "abcdefghij", 1, 4, -4);
    test("abcdefghij", 1, 0, "abcdefghij", 1, 8, -8);
    test("abcdefghij", 1, 0, "abcdefghij", 1, 9, -9);
    test("abcdefghij", 1, 0, "abcdefghij", 1, 10, -9);
    test("abcdefghij", 1, 0, "abcdefghij", 5, 0, 0);
    test("abcdefghij", 1, 0, "abcdefghij", 5, 1, -1);
    test("abcdefghij", 1, 0, "abcdefghij", 5, 2, -2);
    test("abcdefghij", 1, 0, "abcdefghij", 5, 4, -4);
    test("abcdefghij", 1, 0, "abcdefghij", 5, 5, -5);
    test("abcdefghij", 1, 0, "abcdefghij", 5, 6, -5);
    test("abcdefghij", 1, 0, "abcdefghij", 9, 0, 0);
    test("abcdefghij", 1, 0, "abcdefghij", 9, 1, -1);
    test("abcdefghij", 1, 0, "abcdefghij", 9, 2, -1);
    test("abcdefghij", 1, 0, "abcdefghij", 10, 0, 0);
    test("abcdefghij", 1, 0, "abcdefghij", 10, 1, 0);
    test("abcdefghij", 1, 0, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 0, 0, 0);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 0, 1, -1);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 0, 10, -10);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 0, 19, -19);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 0, 20, -20);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 0, 21, -20);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 1, 0, 0);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 1, 9, -9);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 1, 18, -18);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 1, 19, -19);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 1, 20, -19);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 10, 0, 0);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 10, 1, -1);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 19, 0, 0);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 19, 1, -1);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 19, 2, -1);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 20, 0, 0);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 20, 1, 0);
    test("abcdefghij", 1, 0, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 1, 1, "", 0, 0, 1);
    test("abcdefghij", 1, 1, "", 0, 1, 1);
    test("abcdefghij", 1, 1, "", 1, 0, 0);
    test("abcdefghij", 1, 1, "abcde", 0, 0, 1);
    test("abcdefghij", 1, 1, "abcde", 0, 1, 1);
    test("abcdefghij", 1, 1, "abcde", 0, 2, 1);
    test("abcdefghij", 1, 1, "abcde", 0, 4, 1);
    test("abcdefghij", 1, 1, "abcde", 0, 5, 1);
    test("abcdefghij", 1, 1, "abcde", 0, 6, 1);
    test("abcdefghij", 1, 1, "abcde", 1, 0, 1);
    test("abcdefghij", 1, 1, "abcde", 1, 1, 0);
    test("abcdefghij", 1, 1, "abcde", 1, 2, -1);
    test("abcdefghij", 1, 1, "abcde", 1, 3, -2);
    test("abcdefghij", 1, 1, "abcde", 1, 4, -3);
    test("abcdefghij", 1, 1, "abcde", 1, 5, -3);
    test("abcdefghij", 1, 1, "abcde", 2, 0, 1);
    test("abcdefghij", 1, 1, "abcde", 2, 1, -1);
    test("abcdefghij", 1, 1, "abcde", 2, 2, -1);
    test("abcdefghij", 1, 1, "abcde", 2, 3, -1);
    test("abcdefghij", 1, 1, "abcde", 2, 4, -1);
    test("abcdefghij", 1, 1, "abcde", 4, 0, 1);
    test("abcdefghij", 1, 1, "abcde", 4, 1, -3);
    test("abcdefghij", 1, 1, "abcde", 4, 2, -3);
    test("abcdefghij", 1, 1, "abcde", 5, 0, 1);
    test("abcdefghij", 1, 1, "abcde", 5, 1, 1);
    test("abcdefghij", 1, 1, "abcde", 6, 0, 0);
    test("abcdefghij", 1, 1, "abcdefghij", 0, 0, 1);
    test("abcdefghij", 1, 1, "abcdefghij", 0, 1, 1);
    test("abcdefghij", 1, 1, "abcdefghij", 0, 5, 1);
    test("abcdefghij", 1, 1, "abcdefghij", 0, 9, 1);
    test("abcdefghij", 1, 1, "abcdefghij", 0, 10, 1);
    test("abcdefghij", 1, 1, "abcdefghij", 0, 11, 1);
    test("abcdefghij", 1, 1, "abcdefghij", 1, 0, 1);
    test("abcdefghij", 1, 1, "abcdefghij", 1, 1, 0);
    test("abcdefghij", 1, 1, "abcdefghij", 1, 4, -3);
    test("abcdefghij", 1, 1, "abcdefghij", 1, 8, -7);
    test("abcdefghij", 1, 1, "abcdefghij", 1, 9, -8);
    test("abcdefghij", 1, 1, "abcdefghij", 1, 10, -8);
    test("abcdefghij", 1, 1, "abcdefghij", 5, 0, 1);
    test("abcdefghij", 1, 1, "abcdefghij", 5, 1, -4);
    test("abcdefghij", 1, 1, "abcdefghij", 5, 2, -4);
    test("abcdefghij", 1, 1, "abcdefghij", 5, 4, -4);
    test("abcdefghij", 1, 1, "abcdefghij", 5, 5, -4);
    test("abcdefghij", 1, 1, "abcdefghij", 5, 6, -4);
    test("abcdefghij", 1, 1, "abcdefghij", 9, 0, 1);
    test("abcdefghij", 1, 1, "abcdefghij", 9, 1, -8);
    test("abcdefghij", 1, 1, "abcdefghij", 9, 2, -8);
    test("abcdefghij", 1, 1, "abcdefghij", 10, 0, 1);
    test("abcdefghij", 1, 1, "abcdefghij", 10, 1, 1);
    test("abcdefghij", 1, 1, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 0, 0, 1);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 0, 1, 1);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 0, 10, 1);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 0, 19, 1);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 0, 20, 1);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 0, 21, 1);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 1, 0, 1);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 1, 1, 0);
}

void test25()
{
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 1, 9, -8);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 1, 18, -17);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 1, 19, -18);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 1, 20, -18);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 10, 0, 1);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 10, 1, -9);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 10, 5, -9);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 10, 10, -9);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 10, 11, -9);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 19, 0, 1);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 19, 1, -18);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 19, 2, -18);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 20, 0, 1);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 20, 1, 1);
    test("abcdefghij", 1, 1, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 1, 4, "", 0, 0, 4);
    test("abcdefghij", 1, 4, "", 0, 1, 4);
    test("abcdefghij", 1, 4, "", 1, 0, 0);
    test("abcdefghij", 1, 4, "abcde", 0, 0, 4);
    test("abcdefghij", 1, 4, "abcde", 0, 1, 1);
    test("abcdefghij", 1, 4, "abcde", 0, 2, 1);
    test("abcdefghij", 1, 4, "abcde", 0, 4, 1);
    test("abcdefghij", 1, 4, "abcde", 0, 5, 1);
    test("abcdefghij", 1, 4, "abcde", 0, 6, 1);
    test("abcdefghij", 1, 4, "abcde", 1, 0, 4);
    test("abcdefghij", 1, 4, "abcde", 1, 1, 3);
    test("abcdefghij", 1, 4, "abcde", 1, 2, 2);
    test("abcdefghij", 1, 4, "abcde", 1, 3, 1);
    test("abcdefghij", 1, 4, "abcde", 1, 4, 0);
    test("abcdefghij", 1, 4, "abcde", 1, 5, 0);
    test("abcdefghij", 1, 4, "abcde", 2, 0, 4);
    test("abcdefghij", 1, 4, "abcde", 2, 1, -1);
    test("abcdefghij", 1, 4, "abcde", 2, 2, -1);
    test("abcdefghij", 1, 4, "abcde", 2, 3, -1);
    test("abcdefghij", 1, 4, "abcde", 2, 4, -1);
    test("abcdefghij", 1, 4, "abcde", 4, 0, 4);
    test("abcdefghij", 1, 4, "abcde", 4, 1, -3);
    test("abcdefghij", 1, 4, "abcde", 4, 2, -3);
    test("abcdefghij", 1, 4, "abcde", 5, 0, 4);
    test("abcdefghij", 1, 4, "abcde", 5, 1, 4);
    test("abcdefghij", 1, 4, "abcde", 6, 0, 0);
    test("abcdefghij", 1, 4, "abcdefghij", 0, 0, 4);
    test("abcdefghij", 1, 4, "abcdefghij", 0, 1, 1);
    test("abcdefghij", 1, 4, "abcdefghij", 0, 5, 1);
    test("abcdefghij", 1, 4, "abcdefghij", 0, 9, 1);
    test("abcdefghij", 1, 4, "abcdefghij", 0, 10, 1);
    test("abcdefghij", 1, 4, "abcdefghij", 0, 11, 1);
    test("abcdefghij", 1, 4, "abcdefghij", 1, 0, 4);
    test("abcdefghij", 1, 4, "abcdefghij", 1, 1, 3);
    test("abcdefghij", 1, 4, "abcdefghij", 1, 4, 0);
    test("abcdefghij", 1, 4, "abcdefghij", 1, 8, -4);
    test("abcdefghij", 1, 4, "abcdefghij", 1, 9, -5);
    test("abcdefghij", 1, 4, "abcdefghij", 1, 10, -5);
    test("abcdefghij", 1, 4, "abcdefghij", 5, 0, 4);
    test("abcdefghij", 1, 4, "abcdefghij", 5, 1, -4);
    test("abcdefghij", 1, 4, "abcdefghij", 5, 2, -4);
    test("abcdefghij", 1, 4, "abcdefghij", 5, 4, -4);
    test("abcdefghij", 1, 4, "abcdefghij", 5, 5, -4);
    test("abcdefghij", 1, 4, "abcdefghij", 5, 6, -4);
    test("abcdefghij", 1, 4, "abcdefghij", 9, 0, 4);
    test("abcdefghij", 1, 4, "abcdefghij", 9, 1, -8);
    test("abcdefghij", 1, 4, "abcdefghij", 9, 2, -8);
    test("abcdefghij", 1, 4, "abcdefghij", 10, 0, 4);
    test("abcdefghij", 1, 4, "abcdefghij", 10, 1, 4);
    test("abcdefghij", 1, 4, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 0, 0, 4);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 0, 1, 1);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 0, 10, 1);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 0, 19, 1);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 0, 20, 1);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 0, 21, 1);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 1, 0, 4);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 1, 1, 3);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 1, 9, -5);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 1, 18, -14);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 1, 19, -15);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 1, 20, -15);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 10, 0, 4);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 10, 1, -9);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 10, 5, -9);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 10, 10, -9);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 10, 11, -9);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 19, 0, 4);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 19, 1, -18);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 19, 2, -18);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 20, 0, 4);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 20, 1, 4);
    test("abcdefghij", 1, 4, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 1, 8, "", 0, 0, 8);
    test("abcdefghij", 1, 8, "", 0, 1, 8);
    test("abcdefghij", 1, 8, "", 1, 0, 0);
    test("abcdefghij", 1, 8, "abcde", 0, 0, 8);
    test("abcdefghij", 1, 8, "abcde", 0, 1, 1);
    test("abcdefghij", 1, 8, "abcde", 0, 2, 1);
    test("abcdefghij", 1, 8, "abcde", 0, 4, 1);
    test("abcdefghij", 1, 8, "abcde", 0, 5, 1);
    test("abcdefghij", 1, 8, "abcde", 0, 6, 1);
    test("abcdefghij", 1, 8, "abcde", 1, 0, 8);
}

void test26()
{
    test("abcdefghij", 1, 8, "abcde", 1, 1, 7);
    test("abcdefghij", 1, 8, "abcde", 1, 2, 6);
    test("abcdefghij", 1, 8, "abcde", 1, 3, 5);
    test("abcdefghij", 1, 8, "abcde", 1, 4, 4);
    test("abcdefghij", 1, 8, "abcde", 1, 5, 4);
    test("abcdefghij", 1, 8, "abcde", 2, 0, 8);
    test("abcdefghij", 1, 8, "abcde", 2, 1, -1);
    test("abcdefghij", 1, 8, "abcde", 2, 2, -1);
    test("abcdefghij", 1, 8, "abcde", 2, 3, -1);
    test("abcdefghij", 1, 8, "abcde", 2, 4, -1);
    test("abcdefghij", 1, 8, "abcde", 4, 0, 8);
    test("abcdefghij", 1, 8, "abcde", 4, 1, -3);
    test("abcdefghij", 1, 8, "abcde", 4, 2, -3);
    test("abcdefghij", 1, 8, "abcde", 5, 0, 8);
    test("abcdefghij", 1, 8, "abcde", 5, 1, 8);
    test("abcdefghij", 1, 8, "abcde", 6, 0, 0);
    test("abcdefghij", 1, 8, "abcdefghij", 0, 0, 8);
    test("abcdefghij", 1, 8, "abcdefghij", 0, 1, 1);
    test("abcdefghij", 1, 8, "abcdefghij", 0, 5, 1);
    test("abcdefghij", 1, 8, "abcdefghij", 0, 9, 1);
    test("abcdefghij", 1, 8, "abcdefghij", 0, 10, 1);
    test("abcdefghij", 1, 8, "abcdefghij", 0, 11, 1);
    test("abcdefghij", 1, 8, "abcdefghij", 1, 0, 8);
    test("abcdefghij", 1, 8, "abcdefghij", 1, 1, 7);
    test("abcdefghij", 1, 8, "abcdefghij", 1, 4, 4);
    test("abcdefghij", 1, 8, "abcdefghij", 1, 8, 0);
    test("abcdefghij", 1, 8, "abcdefghij", 1, 9, -1);
    test("abcdefghij", 1, 8, "abcdefghij", 1, 10, -1);
    test("abcdefghij", 1, 8, "abcdefghij", 5, 0, 8);
    test("abcdefghij", 1, 8, "abcdefghij", 5, 1, -4);
    test("abcdefghij", 1, 8, "abcdefghij", 5, 2, -4);
    test("abcdefghij", 1, 8, "abcdefghij", 5, 4, -4);
    test("abcdefghij", 1, 8, "abcdefghij", 5, 5, -4);
    test("abcdefghij", 1, 8, "abcdefghij", 5, 6, -4);
    test("abcdefghij", 1, 8, "abcdefghij", 9, 0, 8);
    test("abcdefghij", 1, 8, "abcdefghij", 9, 1, -8);
    test("abcdefghij", 1, 8, "abcdefghij", 9, 2, -8);
    test("abcdefghij", 1, 8, "abcdefghij", 10, 0, 8);
    test("abcdefghij", 1, 8, "abcdefghij", 10, 1, 8);
    test("abcdefghij", 1, 8, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 0, 0, 8);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 0, 1, 1);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 0, 10, 1);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 0, 19, 1);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 0, 20, 1);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 0, 21, 1);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 1, 0, 8);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 1, 1, 7);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 1, 9, -1);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 1, 18, -10);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 1, 19, -11);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 1, 20, -11);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 10, 0, 8);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 10, 1, -9);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 10, 5, -9);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 10, 10, -9);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 10, 11, -9);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 19, 0, 8);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 19, 1, -18);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 19, 2, -18);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 20, 0, 8);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 20, 1, 8);
    test("abcdefghij", 1, 8, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 1, 9, "", 0, 0, 9);
    test("abcdefghij", 1, 9, "", 0, 1, 9);
    test("abcdefghij", 1, 9, "", 1, 0, 0);
    test("abcdefghij", 1, 9, "abcde", 0, 0, 9);
    test("abcdefghij", 1, 9, "abcde", 0, 1, 1);
    test("abcdefghij", 1, 9, "abcde", 0, 2, 1);
    test("abcdefghij", 1, 9, "abcde", 0, 4, 1);
    test("abcdefghij", 1, 9, "abcde", 0, 5, 1);
    test("abcdefghij", 1, 9, "abcde", 0, 6, 1);
    test("abcdefghij", 1, 9, "abcde", 1, 0, 9);
    test("abcdefghij", 1, 9, "abcde", 1, 1, 8);
    test("abcdefghij", 1, 9, "abcde", 1, 2, 7);
    test("abcdefghij", 1, 9, "abcde", 1, 3, 6);
    test("abcdefghij", 1, 9, "abcde", 1, 4, 5);
    test("abcdefghij", 1, 9, "abcde", 1, 5, 5);
    test("abcdefghij", 1, 9, "abcde", 2, 0, 9);
    test("abcdefghij", 1, 9, "abcde", 2, 1, -1);
    test("abcdefghij", 1, 9, "abcde", 2, 2, -1);
    test("abcdefghij", 1, 9, "abcde", 2, 3, -1);
    test("abcdefghij", 1, 9, "abcde", 2, 4, -1);
    test("abcdefghij", 1, 9, "abcde", 4, 0, 9);
    test("abcdefghij", 1, 9, "abcde", 4, 1, -3);
    test("abcdefghij", 1, 9, "abcde", 4, 2, -3);
    test("abcdefghij", 1, 9, "abcde", 5, 0, 9);
    test("abcdefghij", 1, 9, "abcde", 5, 1, 9);
    test("abcdefghij", 1, 9, "abcde", 6, 0, 0);
    test("abcdefghij", 1, 9, "abcdefghij", 0, 0, 9);
    test("abcdefghij", 1, 9, "abcdefghij", 0, 1, 1);
    test("abcdefghij", 1, 9, "abcdefghij", 0, 5, 1);
    test("abcdefghij", 1, 9, "abcdefghij", 0, 9, 1);
    test("abcdefghij", 1, 9, "abcdefghij", 0, 10, 1);
    test("abcdefghij", 1, 9, "abcdefghij", 0, 11, 1);
    test("abcdefghij", 1, 9, "abcdefghij", 1, 0, 9);
    test("abcdefghij", 1, 9, "abcdefghij", 1, 1, 8);
    test("abcdefghij", 1, 9, "abcdefghij", 1, 4, 5);
    test("abcdefghij", 1, 9, "abcdefghij", 1, 8, 1);
}

void test27()
{
    test("abcdefghij", 1, 9, "abcdefghij", 1, 9, 0);
    test("abcdefghij", 1, 9, "abcdefghij", 1, 10, 0);
    test("abcdefghij", 1, 9, "abcdefghij", 5, 0, 9);
    test("abcdefghij", 1, 9, "abcdefghij", 5, 1, -4);
    test("abcdefghij", 1, 9, "abcdefghij", 5, 2, -4);
    test("abcdefghij", 1, 9, "abcdefghij", 5, 4, -4);
    test("abcdefghij", 1, 9, "abcdefghij", 5, 5, -4);
    test("abcdefghij", 1, 9, "abcdefghij", 5, 6, -4);
    test("abcdefghij", 1, 9, "abcdefghij", 9, 0, 9);
    test("abcdefghij", 1, 9, "abcdefghij", 9, 1, -8);
    test("abcdefghij", 1, 9, "abcdefghij", 9, 2, -8);
    test("abcdefghij", 1, 9, "abcdefghij", 10, 0, 9);
    test("abcdefghij", 1, 9, "abcdefghij", 10, 1, 9);
    test("abcdefghij", 1, 9, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 0, 0, 9);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 0, 1, 1);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 0, 10, 1);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 0, 19, 1);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 0, 20, 1);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 0, 21, 1);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 1, 0, 9);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 1, 1, 8);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 1, 9, 0);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 1, 18, -9);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 1, 19, -10);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 1, 20, -10);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 10, 0, 9);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 10, 1, -9);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 10, 5, -9);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 10, 10, -9);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 10, 11, -9);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 19, 0, 9);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 19, 1, -18);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 19, 2, -18);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 20, 0, 9);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 20, 1, 9);
    test("abcdefghij", 1, 9, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 1, 10, "", 0, 0, 9);
    test("abcdefghij", 1, 10, "", 0, 1, 9);
    test("abcdefghij", 1, 10, "", 1, 0, 0);
    test("abcdefghij", 1, 10, "abcde", 0, 0, 9);
    test("abcdefghij", 1, 10, "abcde", 0, 1, 1);
    test("abcdefghij", 1, 10, "abcde", 0, 2, 1);
    test("abcdefghij", 1, 10, "abcde", 0, 4, 1);
    test("abcdefghij", 1, 10, "abcde", 0, 5, 1);
    test("abcdefghij", 1, 10, "abcde", 0, 6, 1);
    test("abcdefghij", 1, 10, "abcde", 1, 0, 9);
    test("abcdefghij", 1, 10, "abcde", 1, 1, 8);
    test("abcdefghij", 1, 10, "abcde", 1, 2, 7);
    test("abcdefghij", 1, 10, "abcde", 1, 3, 6);
    test("abcdefghij", 1, 10, "abcde", 1, 4, 5);
    test("abcdefghij", 1, 10, "abcde", 1, 5, 5);
    test("abcdefghij", 1, 10, "abcde", 2, 0, 9);
    test("abcdefghij", 1, 10, "abcde", 2, 1, -1);
    test("abcdefghij", 1, 10, "abcde", 2, 2, -1);
    test("abcdefghij", 1, 10, "abcde", 2, 3, -1);
    test("abcdefghij", 1, 10, "abcde", 2, 4, -1);
    test("abcdefghij", 1, 10, "abcde", 4, 0, 9);
    test("abcdefghij", 1, 10, "abcde", 4, 1, -3);
    test("abcdefghij", 1, 10, "abcde", 4, 2, -3);
    test("abcdefghij", 1, 10, "abcde", 5, 0, 9);
    test("abcdefghij", 1, 10, "abcde", 5, 1, 9);
    test("abcdefghij", 1, 10, "abcde", 6, 0, 0);
    test("abcdefghij", 1, 10, "abcdefghij", 0, 0, 9);
    test("abcdefghij", 1, 10, "abcdefghij", 0, 1, 1);
    test("abcdefghij", 1, 10, "abcdefghij", 0, 5, 1);
    test("abcdefghij", 1, 10, "abcdefghij", 0, 9, 1);
    test("abcdefghij", 1, 10, "abcdefghij", 0, 10, 1);
    test("abcdefghij", 1, 10, "abcdefghij", 0, 11, 1);
    test("abcdefghij", 1, 10, "abcdefghij", 1, 0, 9);
    test("abcdefghij", 1, 10, "abcdefghij", 1, 1, 8);
    test("abcdefghij", 1, 10, "abcdefghij", 1, 4, 5);
    test("abcdefghij", 1, 10, "abcdefghij", 1, 8, 1);
    test("abcdefghij", 1, 10, "abcdefghij", 1, 9, 0);
    test("abcdefghij", 1, 10, "abcdefghij", 1, 10, 0);
    test("abcdefghij", 1, 10, "abcdefghij", 5, 0, 9);
    test("abcdefghij", 1, 10, "abcdefghij", 5, 1, -4);
    test("abcdefghij", 1, 10, "abcdefghij", 5, 2, -4);
    test("abcdefghij", 1, 10, "abcdefghij", 5, 4, -4);
    test("abcdefghij", 1, 10, "abcdefghij", 5, 5, -4);
    test("abcdefghij", 1, 10, "abcdefghij", 5, 6, -4);
    test("abcdefghij", 1, 10, "abcdefghij", 9, 0, 9);
    test("abcdefghij", 1, 10, "abcdefghij", 9, 1, -8);
    test("abcdefghij", 1, 10, "abcdefghij", 9, 2, -8);
    test("abcdefghij", 1, 10, "abcdefghij", 10, 0, 9);
    test("abcdefghij", 1, 10, "abcdefghij", 10, 1, 9);
    test("abcdefghij", 1, 10, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 0, 0, 9);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 0, 1, 1);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 0, 10, 1);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 0, 19, 1);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 0, 20, 1);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 0, 21, 1);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 1, 0, 9);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 1, 1, 8);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 1, 9, 0);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 1, 18, -9);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 1, 19, -10);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 1, 20, -10);
}

void test28()
{
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 10, 0, 9);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 10, 1, -9);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 10, 5, -9);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 10, 10, -9);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 10, 11, -9);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 19, 0, 9);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 19, 1, -18);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 19, 2, -18);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 20, 0, 9);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 20, 1, 9);
    test("abcdefghij", 1, 10, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 5, 0, "", 0, 0, 0);
    test("abcdefghij", 5, 0, "", 0, 1, 0);
    test("abcdefghij", 5, 0, "", 1, 0, 0);
    test("abcdefghij", 5, 0, "abcde", 0, 0, 0);
    test("abcdefghij", 5, 0, "abcde", 0, 1, -1);
    test("abcdefghij", 5, 0, "abcde", 0, 2, -2);
    test("abcdefghij", 5, 0, "abcde", 0, 4, -4);
    test("abcdefghij", 5, 0, "abcde", 0, 5, -5);
    test("abcdefghij", 5, 0, "abcde", 0, 6, -5);
    test("abcdefghij", 5, 0, "abcde", 1, 0, 0);
    test("abcdefghij", 5, 0, "abcde", 1, 1, -1);
    test("abcdefghij", 5, 0, "abcde", 1, 2, -2);
    test("abcdefghij", 5, 0, "abcde", 1, 3, -3);
    test("abcdefghij", 5, 0, "abcde", 1, 4, -4);
    test("abcdefghij", 5, 0, "abcde", 1, 5, -4);
    test("abcdefghij", 5, 0, "abcde", 2, 0, 0);
    test("abcdefghij", 5, 0, "abcde", 2, 1, -1);
    test("abcdefghij", 5, 0, "abcde", 2, 2, -2);
    test("abcdefghij", 5, 0, "abcde", 2, 3, -3);
    test("abcdefghij", 5, 0, "abcde", 2, 4, -3);
    test("abcdefghij", 5, 0, "abcde", 4, 0, 0);
    test("abcdefghij", 5, 0, "abcde", 4, 1, -1);
    test("abcdefghij", 5, 0, "abcde", 4, 2, -1);
    test("abcdefghij", 5, 0, "abcde", 5, 0, 0);
    test("abcdefghij", 5, 0, "abcde", 5, 1, 0);
    test("abcdefghij", 5, 0, "abcde", 6, 0, 0);
    test("abcdefghij", 5, 0, "abcdefghij", 0, 0, 0);
    test("abcdefghij", 5, 0, "abcdefghij", 0, 1, -1);
    test("abcdefghij", 5, 0, "abcdefghij", 0, 5, -5);
    test("abcdefghij", 5, 0, "abcdefghij", 0, 9, -9);
    test("abcdefghij", 5, 0, "abcdefghij", 0, 10, -10);
    test("abcdefghij", 5, 0, "abcdefghij", 0, 11, -10);
    test("abcdefghij", 5, 0, "abcdefghij", 1, 0, 0);
    test("abcdefghij", 5, 0, "abcdefghij", 1, 1, -1);
    test("abcdefghij", 5, 0, "abcdefghij", 1, 4, -4);
    test("abcdefghij", 5, 0, "abcdefghij", 1, 8, -8);
    test("abcdefghij", 5, 0, "abcdefghij", 1, 9, -9);
    test("abcdefghij", 5, 0, "abcdefghij", 1, 10, -9);
    test("abcdefghij", 5, 0, "abcdefghij", 5, 0, 0);
    test("abcdefghij", 5, 0, "abcdefghij", 5, 1, -1);
    test("abcdefghij", 5, 0, "abcdefghij", 5, 2, -2);
    test("abcdefghij", 5, 0, "abcdefghij", 5, 4, -4);
    test("abcdefghij", 5, 0, "abcdefghij", 5, 5, -5);
    test("abcdefghij", 5, 0, "abcdefghij", 5, 6, -5);
    test("abcdefghij", 5, 0, "abcdefghij", 9, 0, 0);
    test("abcdefghij", 5, 0, "abcdefghij", 9, 1, -1);
    test("abcdefghij", 5, 0, "abcdefghij", 9, 2, -1);
    test("abcdefghij", 5, 0, "abcdefghij", 10, 0, 0);
    test("abcdefghij", 5, 0, "abcdefghij", 10, 1, 0);
    test("abcdefghij", 5, 0, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 0, 0, 0);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 0, 1, -1);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 0, 10, -10);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 0, 19, -19);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 0, 20, -20);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 0, 21, -20);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 1, 0, 0);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 1, 9, -9);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 1, 18, -18);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 1, 19, -19);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 1, 20, -19);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 10, 0, 0);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 10, 1, -1);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 19, 0, 0);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 19, 1, -1);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 19, 2, -1);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 20, 0, 0);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 20, 1, 0);
    test("abcdefghij", 5, 0, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 5, 1, "", 0, 0, 1);
    test("abcdefghij", 5, 1, "", 0, 1, 1);
    test("abcdefghij", 5, 1, "", 1, 0, 0);
    test("abcdefghij", 5, 1, "abcde", 0, 0, 1);
    test("abcdefghij", 5, 1, "abcde", 0, 1, 5);
    test("abcdefghij", 5, 1, "abcde", 0, 2, 5);
    test("abcdefghij", 5, 1, "abcde", 0, 4, 5);
    test("abcdefghij", 5, 1, "abcde", 0, 5, 5);
    test("abcdefghij", 5, 1, "abcde", 0, 6, 5);
    test("abcdefghij", 5, 1, "abcde", 1, 0, 1);
    test("abcdefghij", 5, 1, "abcde", 1, 1, 4);
    test("abcdefghij", 5, 1, "abcde", 1, 2, 4);
    test("abcdefghij", 5, 1, "abcde", 1, 3, 4);
    test("abcdefghij", 5, 1, "abcde", 1, 4, 4);
}

void test29()
{
    test("abcdefghij", 5, 1, "abcde", 1, 5, 4);
    test("abcdefghij", 5, 1, "abcde", 2, 0, 1);
    test("abcdefghij", 5, 1, "abcde", 2, 1, 3);
    test("abcdefghij", 5, 1, "abcde", 2, 2, 3);
    test("abcdefghij", 5, 1, "abcde", 2, 3, 3);
    test("abcdefghij", 5, 1, "abcde", 2, 4, 3);
    test("abcdefghij", 5, 1, "abcde", 4, 0, 1);
    test("abcdefghij", 5, 1, "abcde", 4, 1, 1);
    test("abcdefghij", 5, 1, "abcde", 4, 2, 1);
    test("abcdefghij", 5, 1, "abcde", 5, 0, 1);
    test("abcdefghij", 5, 1, "abcde", 5, 1, 1);
    test("abcdefghij", 5, 1, "abcde", 6, 0, 0);
    test("abcdefghij", 5, 1, "abcdefghij", 0, 0, 1);
    test("abcdefghij", 5, 1, "abcdefghij", 0, 1, 5);
    test("abcdefghij", 5, 1, "abcdefghij", 0, 5, 5);
    test("abcdefghij", 5, 1, "abcdefghij", 0, 9, 5);
    test("abcdefghij", 5, 1, "abcdefghij", 0, 10, 5);
    test("abcdefghij", 5, 1, "abcdefghij", 0, 11, 5);
    test("abcdefghij", 5, 1, "abcdefghij", 1, 0, 1);
    test("abcdefghij", 5, 1, "abcdefghij", 1, 1, 4);
    test("abcdefghij", 5, 1, "abcdefghij", 1, 4, 4);
    test("abcdefghij", 5, 1, "abcdefghij", 1, 8, 4);
    test("abcdefghij", 5, 1, "abcdefghij", 1, 9, 4);
    test("abcdefghij", 5, 1, "abcdefghij", 1, 10, 4);
    test("abcdefghij", 5, 1, "abcdefghij", 5, 0, 1);
    test("abcdefghij", 5, 1, "abcdefghij", 5, 1, 0);
    test("abcdefghij", 5, 1, "abcdefghij", 5, 2, -1);
    test("abcdefghij", 5, 1, "abcdefghij", 5, 4, -3);
    test("abcdefghij", 5, 1, "abcdefghij", 5, 5, -4);
    test("abcdefghij", 5, 1, "abcdefghij", 5, 6, -4);
    test("abcdefghij", 5, 1, "abcdefghij", 9, 0, 1);
    test("abcdefghij", 5, 1, "abcdefghij", 9, 1, -4);
    test("abcdefghij", 5, 1, "abcdefghij", 9, 2, -4);
    test("abcdefghij", 5, 1, "abcdefghij", 10, 0, 1);
    test("abcdefghij", 5, 1, "abcdefghij", 10, 1, 1);
    test("abcdefghij", 5, 1, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 0, 0, 1);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 0, 1, 5);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 0, 10, 5);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 0, 19, 5);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 0, 20, 5);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 0, 21, 5);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 1, 0, 1);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 1, 1, 4);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 1, 9, 4);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 1, 18, 4);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 1, 19, 4);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 1, 20, 4);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 10, 0, 1);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 10, 1, -5);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 10, 9, -5);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 10, 10, -5);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 10, 11, -5);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 19, 0, 1);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 19, 1, -14);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 19, 2, -14);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 20, 0, 1);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 20, 1, 1);
    test("abcdefghij", 5, 1, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 5, 2, "", 0, 0, 2);
    test("abcdefghij", 5, 2, "", 0, 1, 2);
    test("abcdefghij", 5, 2, "", 1, 0, 0);
    test("abcdefghij", 5, 2, "abcde", 0, 0, 2);
    test("abcdefghij", 5, 2, "abcde", 0, 1, 5);
    test("abcdefghij", 5, 2, "abcde", 0, 2, 5);
    test("abcdefghij", 5, 2, "abcde", 0, 4, 5);
    test("abcdefghij", 5, 2, "abcde", 0, 5, 5);
    test("abcdefghij", 5, 2, "abcde", 0, 6, 5);
    test("abcdefghij", 5, 2, "abcde", 1, 0, 2);
    test("abcdefghij", 5, 2, "abcde", 1, 1, 4);
    test("abcdefghij", 5, 2, "abcde", 1, 2, 4);
    test("abcdefghij", 5, 2, "abcde", 1, 3, 4);
    test("abcdefghij", 5, 2, "abcde", 1, 4, 4);
    test("abcdefghij", 5, 2, "abcde", 1, 5, 4);
    test("abcdefghij", 5, 2, "abcde", 2, 0, 2);
    test("abcdefghij", 5, 2, "abcde", 2, 1, 3);
    test("abcdefghij", 5, 2, "abcde", 2, 2, 3);
    test("abcdefghij", 5, 2, "abcde", 2, 3, 3);
    test("abcdefghij", 5, 2, "abcde", 2, 4, 3);
    test("abcdefghij", 5, 2, "abcde", 4, 0, 2);
    test("abcdefghij", 5, 2, "abcde", 4, 1, 1);
    test("abcdefghij", 5, 2, "abcde", 4, 2, 1);
    test("abcdefghij", 5, 2, "abcde", 5, 0, 2);
    test("abcdefghij", 5, 2, "abcde", 5, 1, 2);
    test("abcdefghij", 5, 2, "abcde", 6, 0, 0);
    test("abcdefghij", 5, 2, "abcdefghij", 0, 0, 2);
    test("abcdefghij", 5, 2, "abcdefghij", 0, 1, 5);
    test("abcdefghij", 5, 2, "abcdefghij", 0, 5, 5);
    test("abcdefghij", 5, 2, "abcdefghij", 0, 9, 5);
    test("abcdefghij", 5, 2, "abcdefghij", 0, 10, 5);
    test("abcdefghij", 5, 2, "abcdefghij", 0, 11, 5);
    test("abcdefghij", 5, 2, "abcdefghij", 1, 0, 2);
    test("abcdefghij", 5, 2, "abcdefghij", 1, 1, 4);
    test("abcdefghij", 5, 2, "abcdefghij", 1, 4, 4);
    test("abcdefghij", 5, 2, "abcdefghij", 1, 8, 4);
    test("abcdefghij", 5, 2, "abcdefghij", 1, 9, 4);
    test("abcdefghij", 5, 2, "abcdefghij", 1, 10, 4);
    test("abcdefghij", 5, 2, "abcdefghij", 5, 0, 2);
    test("abcdefghij", 5, 2, "abcdefghij", 5, 1, 1);
}

void test30()
{
    test("abcdefghij", 5, 2, "abcdefghij", 5, 2, 0);
    test("abcdefghij", 5, 2, "abcdefghij", 5, 4, -2);
    test("abcdefghij", 5, 2, "abcdefghij", 5, 5, -3);
    test("abcdefghij", 5, 2, "abcdefghij", 5, 6, -3);
    test("abcdefghij", 5, 2, "abcdefghij", 9, 0, 2);
    test("abcdefghij", 5, 2, "abcdefghij", 9, 1, -4);
    test("abcdefghij", 5, 2, "abcdefghij", 9, 2, -4);
    test("abcdefghij", 5, 2, "abcdefghij", 10, 0, 2);
    test("abcdefghij", 5, 2, "abcdefghij", 10, 1, 2);
    test("abcdefghij", 5, 2, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 0, 0, 2);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 0, 1, 5);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 0, 10, 5);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 0, 19, 5);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 0, 20, 5);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 0, 21, 5);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 1, 0, 2);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 1, 1, 4);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 1, 9, 4);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 1, 18, 4);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 1, 19, 4);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 1, 20, 4);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 10, 0, 2);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 10, 1, -5);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 10, 9, -5);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 10, 10, -5);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 10, 11, -5);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 19, 0, 2);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 19, 1, -14);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 19, 2, -14);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 20, 0, 2);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 20, 1, 2);
    test("abcdefghij", 5, 2, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 5, 4, "", 0, 0, 4);
    test("abcdefghij", 5, 4, "", 0, 1, 4);
    test("abcdefghij", 5, 4, "", 1, 0, 0);
    test("abcdefghij", 5, 4, "abcde", 0, 0, 4);
    test("abcdefghij", 5, 4, "abcde", 0, 1, 5);
    test("abcdefghij", 5, 4, "abcde", 0, 2, 5);
    test("abcdefghij", 5, 4, "abcde", 0, 4, 5);
    test("abcdefghij", 5, 4, "abcde", 0, 5, 5);
    test("abcdefghij", 5, 4, "abcde", 0, 6, 5);
    test("abcdefghij", 5, 4, "abcde", 1, 0, 4);
    test("abcdefghij", 5, 4, "abcde", 1, 1, 4);
    test("abcdefghij", 5, 4, "abcde", 1, 2, 4);
    test("abcdefghij", 5, 4, "abcde", 1, 3, 4);
    test("abcdefghij", 5, 4, "abcde", 1, 4, 4);
    test("abcdefghij", 5, 4, "abcde", 1, 5, 4);
    test("abcdefghij", 5, 4, "abcde", 2, 0, 4);
    test("abcdefghij", 5, 4, "abcde", 2, 1, 3);
    test("abcdefghij", 5, 4, "abcde", 2, 2, 3);
    test("abcdefghij", 5, 4, "abcde", 2, 3, 3);
    test("abcdefghij", 5, 4, "abcde", 2, 4, 3);
    test("abcdefghij", 5, 4, "abcde", 4, 0, 4);
    test("abcdefghij", 5, 4, "abcde", 4, 1, 1);
    test("abcdefghij", 5, 4, "abcde", 4, 2, 1);
    test("abcdefghij", 5, 4, "abcde", 5, 0, 4);
    test("abcdefghij", 5, 4, "abcde", 5, 1, 4);
    test("abcdefghij", 5, 4, "abcde", 6, 0, 0);
    test("abcdefghij", 5, 4, "abcdefghij", 0, 0, 4);
    test("abcdefghij", 5, 4, "abcdefghij", 0, 1, 5);
    test("abcdefghij", 5, 4, "abcdefghij", 0, 5, 5);
    test("abcdefghij", 5, 4, "abcdefghij", 0, 9, 5);
    test("abcdefghij", 5, 4, "abcdefghij", 0, 10, 5);
    test("abcdefghij", 5, 4, "abcdefghij", 0, 11, 5);
    test("abcdefghij", 5, 4, "abcdefghij", 1, 0, 4);
    test("abcdefghij", 5, 4, "abcdefghij", 1, 1, 4);
    test("abcdefghij", 5, 4, "abcdefghij", 1, 4, 4);
    test("abcdefghij", 5, 4, "abcdefghij", 1, 8, 4);
    test("abcdefghij", 5, 4, "abcdefghij", 1, 9, 4);
    test("abcdefghij", 5, 4, "abcdefghij", 1, 10, 4);
    test("abcdefghij", 5, 4, "abcdefghij", 5, 0, 4);
    test("abcdefghij", 5, 4, "abcdefghij", 5, 1, 3);
    test("abcdefghij", 5, 4, "abcdefghij", 5, 2, 2);
    test("abcdefghij", 5, 4, "abcdefghij", 5, 4, 0);
    test("abcdefghij", 5, 4, "abcdefghij", 5, 5, -1);
    test("abcdefghij", 5, 4, "abcdefghij", 5, 6, -1);
    test("abcdefghij", 5, 4, "abcdefghij", 9, 0, 4);
    test("abcdefghij", 5, 4, "abcdefghij", 9, 1, -4);
    test("abcdefghij", 5, 4, "abcdefghij", 9, 2, -4);
    test("abcdefghij", 5, 4, "abcdefghij", 10, 0, 4);
    test("abcdefghij", 5, 4, "abcdefghij", 10, 1, 4);
    test("abcdefghij", 5, 4, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 0, 0, 4);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 0, 1, 5);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 0, 10, 5);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 0, 19, 5);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 0, 20, 5);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 0, 21, 5);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 1, 0, 4);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 1, 1, 4);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 1, 9, 4);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 1, 18, 4);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 1, 19, 4);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 1, 20, 4);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 10, 0, 4);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 10, 1, -5);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 10, 9, -5);
}

void test31()
{
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 10, 10, -5);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 10, 11, -5);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 19, 0, 4);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 19, 1, -14);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 19, 2, -14);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 20, 0, 4);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 20, 1, 4);
    test("abcdefghij", 5, 4, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 5, 5, "", 0, 0, 5);
    test("abcdefghij", 5, 5, "", 0, 1, 5);
    test("abcdefghij", 5, 5, "", 1, 0, 0);
    test("abcdefghij", 5, 5, "abcde", 0, 0, 5);
    test("abcdefghij", 5, 5, "abcde", 0, 1, 5);
    test("abcdefghij", 5, 5, "abcde", 0, 2, 5);
    test("abcdefghij", 5, 5, "abcde", 0, 4, 5);
    test("abcdefghij", 5, 5, "abcde", 0, 5, 5);
    test("abcdefghij", 5, 5, "abcde", 0, 6, 5);
    test("abcdefghij", 5, 5, "abcde", 1, 0, 5);
    test("abcdefghij", 5, 5, "abcde", 1, 1, 4);
    test("abcdefghij", 5, 5, "abcde", 1, 2, 4);
    test("abcdefghij", 5, 5, "abcde", 1, 3, 4);
    test("abcdefghij", 5, 5, "abcde", 1, 4, 4);
    test("abcdefghij", 5, 5, "abcde", 1, 5, 4);
    test("abcdefghij", 5, 5, "abcde", 2, 0, 5);
    test("abcdefghij", 5, 5, "abcde", 2, 1, 3);
    test("abcdefghij", 5, 5, "abcde", 2, 2, 3);
    test("abcdefghij", 5, 5, "abcde", 2, 3, 3);
    test("abcdefghij", 5, 5, "abcde", 2, 4, 3);
    test("abcdefghij", 5, 5, "abcde", 4, 0, 5);
    test("abcdefghij", 5, 5, "abcde", 4, 1, 1);
    test("abcdefghij", 5, 5, "abcde", 4, 2, 1);
    test("abcdefghij", 5, 5, "abcde", 5, 0, 5);
    test("abcdefghij", 5, 5, "abcde", 5, 1, 5);
    test("abcdefghij", 5, 5, "abcde", 6, 0, 0);
    test("abcdefghij", 5, 5, "abcdefghij", 0, 0, 5);
    test("abcdefghij", 5, 5, "abcdefghij", 0, 1, 5);
    test("abcdefghij", 5, 5, "abcdefghij", 0, 5, 5);
    test("abcdefghij", 5, 5, "abcdefghij", 0, 9, 5);
    test("abcdefghij", 5, 5, "abcdefghij", 0, 10, 5);
    test("abcdefghij", 5, 5, "abcdefghij", 0, 11, 5);
    test("abcdefghij", 5, 5, "abcdefghij", 1, 0, 5);
    test("abcdefghij", 5, 5, "abcdefghij", 1, 1, 4);
    test("abcdefghij", 5, 5, "abcdefghij", 1, 4, 4);
    test("abcdefghij", 5, 5, "abcdefghij", 1, 8, 4);
    test("abcdefghij", 5, 5, "abcdefghij", 1, 9, 4);
    test("abcdefghij", 5, 5, "abcdefghij", 1, 10, 4);
    test("abcdefghij", 5, 5, "abcdefghij", 5, 0, 5);
    test("abcdefghij", 5, 5, "abcdefghij", 5, 1, 4);
    test("abcdefghij", 5, 5, "abcdefghij", 5, 2, 3);
    test("abcdefghij", 5, 5, "abcdefghij", 5, 4, 1);
    test("abcdefghij", 5, 5, "abcdefghij", 5, 5, 0);
    test("abcdefghij", 5, 5, "abcdefghij", 5, 6, 0);
    test("abcdefghij", 5, 5, "abcdefghij", 9, 0, 5);
    test("abcdefghij", 5, 5, "abcdefghij", 9, 1, -4);
    test("abcdefghij", 5, 5, "abcdefghij", 9, 2, -4);
    test("abcdefghij", 5, 5, "abcdefghij", 10, 0, 5);
    test("abcdefghij", 5, 5, "abcdefghij", 10, 1, 5);
    test("abcdefghij", 5, 5, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 0, 0, 5);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 0, 1, 5);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 0, 10, 5);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 0, 19, 5);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 0, 20, 5);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 0, 21, 5);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 1, 0, 5);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 1, 1, 4);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 1, 9, 4);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 1, 18, 4);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 1, 19, 4);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 1, 20, 4);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 10, 0, 5);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 10, 1, -5);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 10, 9, -5);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 10, 10, -5);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 10, 11, -5);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 19, 0, 5);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 19, 1, -14);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 19, 2, -14);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 20, 0, 5);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 20, 1, 5);
    test("abcdefghij", 5, 5, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 5, 6, "", 0, 0, 5);
    test("abcdefghij", 5, 6, "", 0, 1, 5);
    test("abcdefghij", 5, 6, "", 1, 0, 0);
    test("abcdefghij", 5, 6, "abcde", 0, 0, 5);
    test("abcdefghij", 5, 6, "abcde", 0, 1, 5);
    test("abcdefghij", 5, 6, "abcde", 0, 2, 5);
    test("abcdefghij", 5, 6, "abcde", 0, 4, 5);
    test("abcdefghij", 5, 6, "abcde", 0, 5, 5);
    test("abcdefghij", 5, 6, "abcde", 0, 6, 5);
    test("abcdefghij", 5, 6, "abcde", 1, 0, 5);
    test("abcdefghij", 5, 6, "abcde", 1, 1, 4);
    test("abcdefghij", 5, 6, "abcde", 1, 2, 4);
    test("abcdefghij", 5, 6, "abcde", 1, 3, 4);
    test("abcdefghij", 5, 6, "abcde", 1, 4, 4);
    test("abcdefghij", 5, 6, "abcde", 1, 5, 4);
    test("abcdefghij", 5, 6, "abcde", 2, 0, 5);
    test("abcdefghij", 5, 6, "abcde", 2, 1, 3);
    test("abcdefghij", 5, 6, "abcde", 2, 2, 3);
}

void test32()
{
    test("abcdefghij", 5, 6, "abcde", 2, 3, 3);
    test("abcdefghij", 5, 6, "abcde", 2, 4, 3);
    test("abcdefghij", 5, 6, "abcde", 4, 0, 5);
    test("abcdefghij", 5, 6, "abcde", 4, 1, 1);
    test("abcdefghij", 5, 6, "abcde", 4, 2, 1);
    test("abcdefghij", 5, 6, "abcde", 5, 0, 5);
    test("abcdefghij", 5, 6, "abcde", 5, 1, 5);
    test("abcdefghij", 5, 6, "abcde", 6, 0, 0);
    test("abcdefghij", 5, 6, "abcdefghij", 0, 0, 5);
    test("abcdefghij", 5, 6, "abcdefghij", 0, 1, 5);
    test("abcdefghij", 5, 6, "abcdefghij", 0, 5, 5);
    test("abcdefghij", 5, 6, "abcdefghij", 0, 9, 5);
    test("abcdefghij", 5, 6, "abcdefghij", 0, 10, 5);
    test("abcdefghij", 5, 6, "abcdefghij", 0, 11, 5);
    test("abcdefghij", 5, 6, "abcdefghij", 1, 0, 5);
    test("abcdefghij", 5, 6, "abcdefghij", 1, 1, 4);
    test("abcdefghij", 5, 6, "abcdefghij", 1, 4, 4);
    test("abcdefghij", 5, 6, "abcdefghij", 1, 8, 4);
    test("abcdefghij", 5, 6, "abcdefghij", 1, 9, 4);
    test("abcdefghij", 5, 6, "abcdefghij", 1, 10, 4);
    test("abcdefghij", 5, 6, "abcdefghij", 5, 0, 5);
    test("abcdefghij", 5, 6, "abcdefghij", 5, 1, 4);
    test("abcdefghij", 5, 6, "abcdefghij", 5, 2, 3);
    test("abcdefghij", 5, 6, "abcdefghij", 5, 4, 1);
    test("abcdefghij", 5, 6, "abcdefghij", 5, 5, 0);
    test("abcdefghij", 5, 6, "abcdefghij", 5, 6, 0);
    test("abcdefghij", 5, 6, "abcdefghij", 9, 0, 5);
    test("abcdefghij", 5, 6, "abcdefghij", 9, 1, -4);
    test("abcdefghij", 5, 6, "abcdefghij", 9, 2, -4);
    test("abcdefghij", 5, 6, "abcdefghij", 10, 0, 5);
    test("abcdefghij", 5, 6, "abcdefghij", 10, 1, 5);
    test("abcdefghij", 5, 6, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 0, 0, 5);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 0, 1, 5);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 0, 10, 5);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 0, 19, 5);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 0, 20, 5);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 0, 21, 5);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 1, 0, 5);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 1, 1, 4);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 1, 9, 4);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 1, 18, 4);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 1, 19, 4);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 1, 20, 4);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 10, 0, 5);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 10, 1, -5);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 10, 9, -5);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 10, 10, -5);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 10, 11, -5);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 19, 0, 5);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 19, 1, -14);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 19, 2, -14);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 20, 0, 5);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 20, 1, 5);
    test("abcdefghij", 5, 6, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 9, 0, "", 0, 0, 0);
    test("abcdefghij", 9, 0, "", 0, 1, 0);
    test("abcdefghij", 9, 0, "", 1, 0, 0);
    test("abcdefghij", 9, 0, "abcde", 0, 0, 0);
    test("abcdefghij", 9, 0, "abcde", 0, 1, -1);
    test("abcdefghij", 9, 0, "abcde", 0, 2, -2);
    test("abcdefghij", 9, 0, "abcde", 0, 4, -4);
    test("abcdefghij", 9, 0, "abcde", 0, 5, -5);
    test("abcdefghij", 9, 0, "abcde", 0, 6, -5);
    test("abcdefghij", 9, 0, "abcde", 1, 0, 0);
    test("abcdefghij", 9, 0, "abcde", 1, 1, -1);
    test("abcdefghij", 9, 0, "abcde", 1, 2, -2);
    test("abcdefghij", 9, 0, "abcde", 1, 3, -3);
    test("abcdefghij", 9, 0, "abcde", 1, 4, -4);
    test("abcdefghij", 9, 0, "abcde", 1, 5, -4);
    test("abcdefghij", 9, 0, "abcde", 2, 0, 0);
    test("abcdefghij", 9, 0, "abcde", 2, 1, -1);
    test("abcdefghij", 9, 0, "abcde", 2, 2, -2);
    test("abcdefghij", 9, 0, "abcde", 2, 3, -3);
    test("abcdefghij", 9, 0, "abcde", 2, 4, -3);
    test("abcdefghij", 9, 0, "abcde", 4, 0, 0);
    test("abcdefghij", 9, 0, "abcde", 4, 1, -1);
    test("abcdefghij", 9, 0, "abcde", 4, 2, -1);
    test("abcdefghij", 9, 0, "abcde", 5, 0, 0);
    test("abcdefghij", 9, 0, "abcde", 5, 1, 0);
    test("abcdefghij", 9, 0, "abcde", 6, 0, 0);
    test("abcdefghij", 9, 0, "abcdefghij", 0, 0, 0);
    test("abcdefghij", 9, 0, "abcdefghij", 0, 1, -1);
    test("abcdefghij", 9, 0, "abcdefghij", 0, 5, -5);
    test("abcdefghij", 9, 0, "abcdefghij", 0, 9, -9);
    test("abcdefghij", 9, 0, "abcdefghij", 0, 10, -10);
    test("abcdefghij", 9, 0, "abcdefghij", 0, 11, -10);
    test("abcdefghij", 9, 0, "abcdefghij", 1, 0, 0);
    test("abcdefghij", 9, 0, "abcdefghij", 1, 1, -1);
    test("abcdefghij", 9, 0, "abcdefghij", 1, 4, -4);
    test("abcdefghij", 9, 0, "abcdefghij", 1, 8, -8);
    test("abcdefghij", 9, 0, "abcdefghij", 1, 9, -9);
    test("abcdefghij", 9, 0, "abcdefghij", 1, 10, -9);
    test("abcdefghij", 9, 0, "abcdefghij", 5, 0, 0);
    test("abcdefghij", 9, 0, "abcdefghij", 5, 1, -1);
    test("abcdefghij", 9, 0, "abcdefghij", 5, 2, -2);
    test("abcdefghij", 9, 0, "abcdefghij", 5, 4, -4);
    test("abcdefghij", 9, 0, "abcdefghij", 5, 5, -5);
    test("abcdefghij", 9, 0, "abcdefghij", 5, 6, -5);
}

void test33()
{
    test("abcdefghij", 9, 0, "abcdefghij", 9, 0, 0);
    test("abcdefghij", 9, 0, "abcdefghij", 9, 1, -1);
    test("abcdefghij", 9, 0, "abcdefghij", 9, 2, -1);
    test("abcdefghij", 9, 0, "abcdefghij", 10, 0, 0);
    test("abcdefghij", 9, 0, "abcdefghij", 10, 1, 0);
    test("abcdefghij", 9, 0, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 0, 0, 0);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 0, 1, -1);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 0, 10, -10);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 0, 19, -19);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 0, 20, -20);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 0, 21, -20);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 1, 0, 0);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 1, 9, -9);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 1, 18, -18);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 1, 19, -19);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 1, 20, -19);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 10, 0, 0);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 10, 1, -1);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 19, 0, 0);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 19, 1, -1);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 19, 2, -1);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 20, 0, 0);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 20, 1, 0);
    test("abcdefghij", 9, 0, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 9, 1, "", 0, 0, 1);
    test("abcdefghij", 9, 1, "", 0, 1, 1);
    test("abcdefghij", 9, 1, "", 1, 0, 0);
    test("abcdefghij", 9, 1, "abcde", 0, 0, 1);
    test("abcdefghij", 9, 1, "abcde", 0, 1, 9);
    test("abcdefghij", 9, 1, "abcde", 0, 2, 9);
    test("abcdefghij", 9, 1, "abcde", 0, 4, 9);
    test("abcdefghij", 9, 1, "abcde", 0, 5, 9);
    test("abcdefghij", 9, 1, "abcde", 0, 6, 9);
    test("abcdefghij", 9, 1, "abcde", 1, 0, 1);
    test("abcdefghij", 9, 1, "abcde", 1, 1, 8);
    test("abcdefghij", 9, 1, "abcde", 1, 2, 8);
    test("abcdefghij", 9, 1, "abcde", 1, 3, 8);
    test("abcdefghij", 9, 1, "abcde", 1, 4, 8);
    test("abcdefghij", 9, 1, "abcde", 1, 5, 8);
    test("abcdefghij", 9, 1, "abcde", 2, 0, 1);
    test("abcdefghij", 9, 1, "abcde", 2, 1, 7);
    test("abcdefghij", 9, 1, "abcde", 2, 2, 7);
    test("abcdefghij", 9, 1, "abcde", 2, 3, 7);
    test("abcdefghij", 9, 1, "abcde", 2, 4, 7);
    test("abcdefghij", 9, 1, "abcde", 4, 0, 1);
    test("abcdefghij", 9, 1, "abcde", 4, 1, 5);
    test("abcdefghij", 9, 1, "abcde", 4, 2, 5);
    test("abcdefghij", 9, 1, "abcde", 5, 0, 1);
    test("abcdefghij", 9, 1, "abcde", 5, 1, 1);
    test("abcdefghij", 9, 1, "abcde", 6, 0, 0);
    test("abcdefghij", 9, 1, "abcdefghij", 0, 0, 1);
    test("abcdefghij", 9, 1, "abcdefghij", 0, 1, 9);
    test("abcdefghij", 9, 1, "abcdefghij", 0, 5, 9);
    test("abcdefghij", 9, 1, "abcdefghij", 0, 9, 9);
    test("abcdefghij", 9, 1, "abcdefghij", 0, 10, 9);
    test("abcdefghij", 9, 1, "abcdefghij", 0, 11, 9);
    test("abcdefghij", 9, 1, "abcdefghij", 1, 0, 1);
    test("abcdefghij", 9, 1, "abcdefghij", 1, 1, 8);
    test("abcdefghij", 9, 1, "abcdefghij", 1, 4, 8);
    test("abcdefghij", 9, 1, "abcdefghij", 1, 8, 8);
    test("abcdefghij", 9, 1, "abcdefghij", 1, 9, 8);
    test("abcdefghij", 9, 1, "abcdefghij", 1, 10, 8);
    test("abcdefghij", 9, 1, "abcdefghij", 5, 0, 1);
    test("abcdefghij", 9, 1, "abcdefghij", 5, 1, 4);
    test("abcdefghij", 9, 1, "abcdefghij", 5, 2, 4);
    test("abcdefghij", 9, 1, "abcdefghij", 5, 4, 4);
    test("abcdefghij", 9, 1, "abcdefghij", 5, 5, 4);
    test("abcdefghij", 9, 1, "abcdefghij", 5, 6, 4);
    test("abcdefghij", 9, 1, "abcdefghij", 9, 0, 1);
    test("abcdefghij", 9, 1, "abcdefghij", 9, 1, 0);
    test("abcdefghij", 9, 1, "abcdefghij", 9, 2, 0);
    test("abcdefghij", 9, 1, "abcdefghij", 10, 0, 1);
    test("abcdefghij", 9, 1, "abcdefghij", 10, 1, 1);
    test("abcdefghij", 9, 1, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 0, 0, 1);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 0, 1, 9);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 0, 10, 9);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 0, 19, 9);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 0, 20, 9);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 0, 21, 9);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 1, 0, 1);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 1, 1, 8);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 1, 9, 8);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 1, 18, 8);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 1, 19, 8);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 1, 20, 8);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 10, 0, 1);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 10, 1, -1);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 10, 5, -1);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 10, 9, -1);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 10, 10, -1);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 10, 11, -1);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 19, 0, 1);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 19, 1, -10);
}

void test34()
{
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 19, 2, -10);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 20, 0, 1);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 20, 1, 1);
    test("abcdefghij", 9, 1, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 9, 2, "", 0, 0, 1);
    test("abcdefghij", 9, 2, "", 0, 1, 1);
    test("abcdefghij", 9, 2, "", 1, 0, 0);
    test("abcdefghij", 9, 2, "abcde", 0, 0, 1);
    test("abcdefghij", 9, 2, "abcde", 0, 1, 9);
    test("abcdefghij", 9, 2, "abcde", 0, 2, 9);
    test("abcdefghij", 9, 2, "abcde", 0, 4, 9);
    test("abcdefghij", 9, 2, "abcde", 0, 5, 9);
    test("abcdefghij", 9, 2, "abcde", 0, 6, 9);
    test("abcdefghij", 9, 2, "abcde", 1, 0, 1);
    test("abcdefghij", 9, 2, "abcde", 1, 1, 8);
    test("abcdefghij", 9, 2, "abcde", 1, 2, 8);
    test("abcdefghij", 9, 2, "abcde", 1, 3, 8);
    test("abcdefghij", 9, 2, "abcde", 1, 4, 8);
    test("abcdefghij", 9, 2, "abcde", 1, 5, 8);
    test("abcdefghij", 9, 2, "abcde", 2, 0, 1);
    test("abcdefghij", 9, 2, "abcde", 2, 1, 7);
    test("abcdefghij", 9, 2, "abcde", 2, 2, 7);
    test("abcdefghij", 9, 2, "abcde", 2, 3, 7);
    test("abcdefghij", 9, 2, "abcde", 2, 4, 7);
    test("abcdefghij", 9, 2, "abcde", 4, 0, 1);
    test("abcdefghij", 9, 2, "abcde", 4, 1, 5);
    test("abcdefghij", 9, 2, "abcde", 4, 2, 5);
    test("abcdefghij", 9, 2, "abcde", 5, 0, 1);
    test("abcdefghij", 9, 2, "abcde", 5, 1, 1);
    test("abcdefghij", 9, 2, "abcde", 6, 0, 0);
    test("abcdefghij", 9, 2, "abcdefghij", 0, 0, 1);
    test("abcdefghij", 9, 2, "abcdefghij", 0, 1, 9);
    test("abcdefghij", 9, 2, "abcdefghij", 0, 5, 9);
    test("abcdefghij", 9, 2, "abcdefghij", 0, 9, 9);
    test("abcdefghij", 9, 2, "abcdefghij", 0, 10, 9);
    test("abcdefghij", 9, 2, "abcdefghij", 0, 11, 9);
    test("abcdefghij", 9, 2, "abcdefghij", 1, 0, 1);
    test("abcdefghij", 9, 2, "abcdefghij", 1, 1, 8);
    test("abcdefghij", 9, 2, "abcdefghij", 1, 4, 8);
    test("abcdefghij", 9, 2, "abcdefghij", 1, 8, 8);
    test("abcdefghij", 9, 2, "abcdefghij", 1, 9, 8);
    test("abcdefghij", 9, 2, "abcdefghij", 1, 10, 8);
    test("abcdefghij", 9, 2, "abcdefghij", 5, 0, 1);
    test("abcdefghij", 9, 2, "abcdefghij", 5, 1, 4);
    test("abcdefghij", 9, 2, "abcdefghij", 5, 2, 4);
    test("abcdefghij", 9, 2, "abcdefghij", 5, 4, 4);
    test("abcdefghij", 9, 2, "abcdefghij", 5, 5, 4);
    test("abcdefghij", 9, 2, "abcdefghij", 5, 6, 4);
    test("abcdefghij", 9, 2, "abcdefghij", 9, 0, 1);
    test("abcdefghij", 9, 2, "abcdefghij", 9, 1, 0);
    test("abcdefghij", 9, 2, "abcdefghij", 9, 2, 0);
    test("abcdefghij", 9, 2, "abcdefghij", 10, 0, 1);
    test("abcdefghij", 9, 2, "abcdefghij", 10, 1, 1);
    test("abcdefghij", 9, 2, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 0, 0, 1);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 0, 1, 9);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 0, 10, 9);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 0, 19, 9);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 0, 20, 9);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 0, 21, 9);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 1, 0, 1);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 1, 1, 8);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 1, 9, 8);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 1, 18, 8);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 1, 19, 8);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 1, 20, 8);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 10, 0, 1);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 10, 1, -1);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 10, 5, -1);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 10, 9, -1);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 10, 10, -1);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 10, 11, -1);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 19, 0, 1);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 19, 1, -10);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 19, 2, -10);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 20, 0, 1);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 20, 1, 1);
    test("abcdefghij", 9, 2, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 10, 0, "", 0, 0, 0);
    test("abcdefghij", 10, 0, "", 0, 1, 0);
    test("abcdefghij", 10, 0, "", 1, 0, 0);
    test("abcdefghij", 10, 0, "abcde", 0, 0, 0);
    test("abcdefghij", 10, 0, "abcde", 0, 1, -1);
    test("abcdefghij", 10, 0, "abcde", 0, 2, -2);
    test("abcdefghij", 10, 0, "abcde", 0, 4, -4);
    test("abcdefghij", 10, 0, "abcde", 0, 5, -5);
    test("abcdefghij", 10, 0, "abcde", 0, 6, -5);
    test("abcdefghij", 10, 0, "abcde", 1, 0, 0);
    test("abcdefghij", 10, 0, "abcde", 1, 1, -1);
    test("abcdefghij", 10, 0, "abcde", 1, 2, -2);
    test("abcdefghij", 10, 0, "abcde", 1, 3, -3);
    test("abcdefghij", 10, 0, "abcde", 1, 4, -4);
    test("abcdefghij", 10, 0, "abcde", 1, 5, -4);
    test("abcdefghij", 10, 0, "abcde", 2, 0, 0);
    test("abcdefghij", 10, 0, "abcde", 2, 1, -1);
    test("abcdefghij", 10, 0, "abcde", 2, 2, -2);
    test("abcdefghij", 10, 0, "abcde", 2, 3, -3);
    test("abcdefghij", 10, 0, "abcde", 2, 4, -3);
    test("abcdefghij", 10, 0, "abcde", 4, 0, 0);
    test("abcdefghij", 10, 0, "abcde", 4, 1, -1);
}

void test35()
{
    test("abcdefghij", 10, 0, "abcde", 4, 2, -1);
    test("abcdefghij", 10, 0, "abcde", 5, 0, 0);
    test("abcdefghij", 10, 0, "abcde", 5, 1, 0);
    test("abcdefghij", 10, 0, "abcde", 6, 0, 0);
    test("abcdefghij", 10, 0, "abcdefghij", 0, 0, 0);
    test("abcdefghij", 10, 0, "abcdefghij", 0, 1, -1);
    test("abcdefghij", 10, 0, "abcdefghij", 0, 5, -5);
    test("abcdefghij", 10, 0, "abcdefghij", 0, 9, -9);
    test("abcdefghij", 10, 0, "abcdefghij", 0, 10, -10);
    test("abcdefghij", 10, 0, "abcdefghij", 0, 11, -10);
    test("abcdefghij", 10, 0, "abcdefghij", 1, 0, 0);
    test("abcdefghij", 10, 0, "abcdefghij", 1, 1, -1);
    test("abcdefghij", 10, 0, "abcdefghij", 1, 4, -4);
    test("abcdefghij", 10, 0, "abcdefghij", 1, 8, -8);
    test("abcdefghij", 10, 0, "abcdefghij", 1, 9, -9);
    test("abcdefghij", 10, 0, "abcdefghij", 1, 10, -9);
    test("abcdefghij", 10, 0, "abcdefghij", 5, 0, 0);
    test("abcdefghij", 10, 0, "abcdefghij", 5, 1, -1);
    test("abcdefghij", 10, 0, "abcdefghij", 5, 2, -2);
    test("abcdefghij", 10, 0, "abcdefghij", 5, 4, -4);
    test("abcdefghij", 10, 0, "abcdefghij", 5, 5, -5);
    test("abcdefghij", 10, 0, "abcdefghij", 5, 6, -5);
    test("abcdefghij", 10, 0, "abcdefghij", 9, 0, 0);
    test("abcdefghij", 10, 0, "abcdefghij", 9, 1, -1);
    test("abcdefghij", 10, 0, "abcdefghij", 9, 2, -1);
    test("abcdefghij", 10, 0, "abcdefghij", 10, 0, 0);
    test("abcdefghij", 10, 0, "abcdefghij", 10, 1, 0);
    test("abcdefghij", 10, 0, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 0, 0, 0);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 0, 1, -1);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 0, 10, -10);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 0, 19, -19);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 0, 20, -20);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 0, 21, -20);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 1, 0, 0);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 1, 9, -9);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 1, 18, -18);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 1, 19, -19);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 1, 20, -19);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 10, 0, 0);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 10, 1, -1);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 19, 0, 0);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 19, 1, -1);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 19, 2, -1);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 20, 0, 0);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 20, 1, 0);
    test("abcdefghij", 10, 0, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 10, 1, "", 0, 0, 0);
    test("abcdefghij", 10, 1, "", 0, 1, 0);
    test("abcdefghij", 10, 1, "", 1, 0, 0);
    test("abcdefghij", 10, 1, "abcde", 0, 0, 0);
    test("abcdefghij", 10, 1, "abcde", 0, 1, -1);
    test("abcdefghij", 10, 1, "abcde", 0, 2, -2);
    test("abcdefghij", 10, 1, "abcde", 0, 4, -4);
    test("abcdefghij", 10, 1, "abcde", 0, 5, -5);
    test("abcdefghij", 10, 1, "abcde", 0, 6, -5);
    test("abcdefghij", 10, 1, "abcde", 1, 0, 0);
    test("abcdefghij", 10, 1, "abcde", 1, 1, -1);
    test("abcdefghij", 10, 1, "abcde", 1, 2, -2);
    test("abcdefghij", 10, 1, "abcde", 1, 3, -3);
    test("abcdefghij", 10, 1, "abcde", 1, 4, -4);
    test("abcdefghij", 10, 1, "abcde", 1, 5, -4);
    test("abcdefghij", 10, 1, "abcde", 2, 0, 0);
    test("abcdefghij", 10, 1, "abcde", 2, 1, -1);
    test("abcdefghij", 10, 1, "abcde", 2, 2, -2);
    test("abcdefghij", 10, 1, "abcde", 2, 3, -3);
    test("abcdefghij", 10, 1, "abcde", 2, 4, -3);
    test("abcdefghij", 10, 1, "abcde", 4, 0, 0);
    test("abcdefghij", 10, 1, "abcde", 4, 1, -1);
    test("abcdefghij", 10, 1, "abcde", 4, 2, -1);
    test("abcdefghij", 10, 1, "abcde", 5, 0, 0);
    test("abcdefghij", 10, 1, "abcde", 5, 1, 0);
    test("abcdefghij", 10, 1, "abcde", 6, 0, 0);
    test("abcdefghij", 10, 1, "abcdefghij", 0, 0, 0);
    test("abcdefghij", 10, 1, "abcdefghij", 0, 1, -1);
    test("abcdefghij", 10, 1, "abcdefghij", 0, 5, -5);
    test("abcdefghij", 10, 1, "abcdefghij", 0, 9, -9);
    test("abcdefghij", 10, 1, "abcdefghij", 0, 10, -10);
    test("abcdefghij", 10, 1, "abcdefghij", 0, 11, -10);
    test("abcdefghij", 10, 1, "abcdefghij", 1, 0, 0);
    test("abcdefghij", 10, 1, "abcdefghij", 1, 1, -1);
    test("abcdefghij", 10, 1, "abcdefghij", 1, 4, -4);
    test("abcdefghij", 10, 1, "abcdefghij", 1, 8, -8);
    test("abcdefghij", 10, 1, "abcdefghij", 1, 9, -9);
    test("abcdefghij", 10, 1, "abcdefghij", 1, 10, -9);
    test("abcdefghij", 10, 1, "abcdefghij", 5, 0, 0);
    test("abcdefghij", 10, 1, "abcdefghij", 5, 1, -1);
    test("abcdefghij", 10, 1, "abcdefghij", 5, 2, -2);
    test("abcdefghij", 10, 1, "abcdefghij", 5, 4, -4);
    test("abcdefghij", 10, 1, "abcdefghij", 5, 5, -5);
    test("abcdefghij", 10, 1, "abcdefghij", 5, 6, -5);
    test("abcdefghij", 10, 1, "abcdefghij", 9, 0, 0);
    test("abcdefghij", 10, 1, "abcdefghij", 9, 1, -1);
    test("abcdefghij", 10, 1, "abcdefghij", 9, 2, -1);
    test("abcdefghij", 10, 1, "abcdefghij", 10, 0, 0);
}

void test36()
{
    test("abcdefghij", 10, 1, "abcdefghij", 10, 1, 0);
    test("abcdefghij", 10, 1, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 0, 0, 0);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 0, 1, -1);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 0, 10, -10);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 0, 19, -19);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 0, 20, -20);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 0, 21, -20);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 1, 0, 0);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 1, 9, -9);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 1, 18, -18);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 1, 19, -19);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 1, 20, -19);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 10, 0, 0);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 10, 1, -1);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 19, 0, 0);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 19, 1, -1);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 19, 2, -1);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 20, 0, 0);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 20, 1, 0);
    test("abcdefghij", 10, 1, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghij", 11, 0, "", 0, 0, 0);
    test("abcdefghij", 11, 0, "", 0, 1, 0);
    test("abcdefghij", 11, 0, "", 1, 0, 0);
    test("abcdefghij", 11, 0, "abcde", 0, 0, 0);
    test("abcdefghij", 11, 0, "abcde", 0, 1, 0);
    test("abcdefghij", 11, 0, "abcde", 0, 2, 0);
    test("abcdefghij", 11, 0, "abcde", 0, 4, 0);
    test("abcdefghij", 11, 0, "abcde", 0, 5, 0);
    test("abcdefghij", 11, 0, "abcde", 0, 6, 0);
    test("abcdefghij", 11, 0, "abcde", 1, 0, 0);
    test("abcdefghij", 11, 0, "abcde", 1, 1, 0);
    test("abcdefghij", 11, 0, "abcde", 1, 2, 0);
    test("abcdefghij", 11, 0, "abcde", 1, 3, 0);
    test("abcdefghij", 11, 0, "abcde", 1, 4, 0);
    test("abcdefghij", 11, 0, "abcde", 1, 5, 0);
    test("abcdefghij", 11, 0, "abcde", 2, 0, 0);
    test("abcdefghij", 11, 0, "abcde", 2, 1, 0);
    test("abcdefghij", 11, 0, "abcde", 2, 2, 0);
    test("abcdefghij", 11, 0, "abcde", 2, 3, 0);
    test("abcdefghij", 11, 0, "abcde", 2, 4, 0);
    test("abcdefghij", 11, 0, "abcde", 4, 0, 0);
    test("abcdefghij", 11, 0, "abcde", 4, 1, 0);
    test("abcdefghij", 11, 0, "abcde", 4, 2, 0);
    test("abcdefghij", 11, 0, "abcde", 5, 0, 0);
    test("abcdefghij", 11, 0, "abcde", 5, 1, 0);
    test("abcdefghij", 11, 0, "abcde", 6, 0, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 0, 0, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 0, 1, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 0, 5, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 0, 9, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 0, 10, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 0, 11, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 1, 0, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 1, 1, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 1, 4, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 1, 8, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 1, 9, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 1, 10, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 5, 0, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 5, 1, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 5, 2, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 5, 4, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 5, 5, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 5, 6, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 9, 0, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 9, 1, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 9, 2, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 10, 0, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 10, 1, 0);
    test("abcdefghij", 11, 0, "abcdefghij", 11, 0, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 0, 0, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 0, 1, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 0, 10, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 0, 19, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 0, 20, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 0, 21, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 1, 0, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 1, 1, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 1, 9, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 1, 18, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 1, 19, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 1, 20, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 10, 0, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 10, 1, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 10, 5, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 10, 9, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 10, 10, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 10, 11, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 19, 0, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 19, 1, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 19, 2, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 20, 0, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 20, 1, 0);
    test("abcdefghij", 11, 0, "abcdefghijklmnopqrst", 21, 0, 0);
}

void test37()
{
    test("abcdefghijklmnopqrst", 0, 0, "", 0, 0, 0);
    test("abcdefghijklmnopqrst", 0, 0, "", 0, 1, 0);
    test("abcdefghijklmnopqrst", 0, 0, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 0, 0, 0);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 0, 1, -1);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 0, 2, -2);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 0, 4, -4);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 0, 5, -5);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 0, 6, -5);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 1, 0, 0);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 1, 1, -1);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 1, 2, -2);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 1, 3, -3);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 1, 4, -4);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 1, 5, -4);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 2, 0, 0);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 2, 1, -1);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 2, 2, -2);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 2, 3, -3);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 2, 4, -3);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 4, 0, 0);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 4, 1, -1);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 4, 2, -1);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 5, 0, 0);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 5, 1, 0);
    test("abcdefghijklmnopqrst", 0, 0, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 0, 0, 0);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 0, 1, -1);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 0, 5, -5);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 0, 9, -9);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 0, 10, -10);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 0, 11, -10);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 1, 0, 0);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 1, 1, -1);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 1, 4, -4);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 1, 8, -8);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 1, 9, -9);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 1, 10, -9);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 5, 0, 0);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 5, 1, -1);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 5, 2, -2);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 5, 4, -4);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 5, 5, -5);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 5, 6, -5);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 9, 0, 0);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 9, 1, -1);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 9, 2, -1);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 10, 0, 0);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 10, 1, 0);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 0, 0, 0);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 0, 1, -1);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 0, 10, -10);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 0, 19, -19);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 0, 20, -20);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 0, 21, -20);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 1, 0, 0);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 1, 9, -9);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 1, 18, -18);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 1, 19, -19);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 1, 20, -19);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 10, 0, 0);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 10, 1, -1);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 19, 0, 0);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 19, 1, -1);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 19, 2, -1);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 20, 0, 0);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 20, 1, 0);
    test("abcdefghijklmnopqrst", 0, 0, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 0, 1, "", 0, 0, 1);
    test("abcdefghijklmnopqrst", 0, 1, "", 0, 1, 1);
    test("abcdefghijklmnopqrst", 0, 1, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 0, 0, 1);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 0, 1, 0);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 0, 2, -1);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 0, 4, -3);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 0, 5, -4);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 0, 6, -4);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 1, 0, 1);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 1, 1, -1);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 1, 2, -1);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 1, 3, -1);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 1, 4, -1);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 1, 5, -1);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 2, 0, 1);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 2, 1, -2);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 2, 2, -2);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 2, 3, -2);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 2, 4, -2);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 4, 0, 1);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 4, 1, -4);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 4, 2, -4);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 5, 0, 1);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 5, 1, 1);
    test("abcdefghijklmnopqrst", 0, 1, "abcde", 6, 0, 0);
}

void test38()
{
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 0, 0, 1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 0, 1, 0);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 0, 5, -4);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 0, 9, -8);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 0, 10, -9);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 0, 11, -9);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 1, 0, 1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 1, 1, -1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 1, 4, -1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 1, 8, -1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 1, 9, -1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 1, 10, -1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 5, 0, 1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 5, 1, -5);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 5, 2, -5);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 5, 4, -5);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 5, 5, -5);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 5, 6, -5);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 9, 0, 1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 9, 1, -9);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 9, 2, -9);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 10, 0, 1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 10, 1, 1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 0, 0, 1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 0, 1, 0);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 0, 10, -9);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 0, 19, -18);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 0, 20, -19);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 0, 21, -19);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 1, 0, 1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 1, 9, -1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 1, 18, -1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 1, 19, -1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 1, 20, -1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 10, 0, 1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 10, 1, -10);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 10, 5, -10);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 10, 9, -10);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 19, 0, 1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 19, 1, -19);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 19, 2, -19);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 20, 0, 1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 20, 1, 1);
    test("abcdefghijklmnopqrst", 0, 1, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 0, 10, "", 0, 0, 10);
    test("abcdefghijklmnopqrst", 0, 10, "", 0, 1, 10);
    test("abcdefghijklmnopqrst", 0, 10, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 0, 0, 10);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 0, 1, 9);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 0, 2, 8);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 0, 4, 6);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 0, 5, 5);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 0, 6, 5);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 1, 0, 10);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 1, 1, -1);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 1, 2, -1);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 1, 3, -1);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 1, 4, -1);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 1, 5, -1);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 2, 0, 10);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 2, 1, -2);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 2, 2, -2);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 2, 3, -2);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 2, 4, -2);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 4, 0, 10);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 4, 1, -4);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 4, 2, -4);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 5, 0, 10);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 5, 1, 10);
    test("abcdefghijklmnopqrst", 0, 10, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 0, 0, 10);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 0, 1, 9);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 0, 5, 5);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 0, 9, 1);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 0, 10, 0);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 0, 11, 0);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 1, 0, 10);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 1, 1, -1);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 1, 4, -1);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 1, 8, -1);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 1, 9, -1);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 1, 10, -1);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 5, 0, 10);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 5, 1, -5);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 5, 2, -5);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 5, 4, -5);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 5, 5, -5);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 5, 6, -5);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 9, 0, 10);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 9, 1, -9);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 9, 2, -9);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 10, 0, 10);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 10, 1, 10);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 0, 0, 10);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 0, 1, 9);
}

void test39()
{
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 0, 10, 0);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 0, 19, -9);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 0, 20, -10);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 0, 21, -10);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 1, 0, 10);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 1, 9, -1);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 1, 18, -1);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 1, 19, -1);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 1, 20, -1);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 10, 0, 10);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 10, 1, -10);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 10, 5, -10);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 10, 9, -10);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 19, 0, 10);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 19, 1, -19);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 19, 2, -19);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 20, 0, 10);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 20, 1, 10);
    test("abcdefghijklmnopqrst", 0, 10, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 0, 19, "", 0, 0, 19);
    test("abcdefghijklmnopqrst", 0, 19, "", 0, 1, 19);
    test("abcdefghijklmnopqrst", 0, 19, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 0, 0, 19);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 0, 1, 18);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 0, 2, 17);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 0, 4, 15);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 0, 5, 14);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 0, 6, 14);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 1, 0, 19);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 1, 1, -1);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 1, 2, -1);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 1, 3, -1);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 1, 4, -1);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 1, 5, -1);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 2, 0, 19);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 2, 1, -2);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 2, 2, -2);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 2, 3, -2);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 2, 4, -2);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 4, 0, 19);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 4, 1, -4);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 4, 2, -4);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 5, 0, 19);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 5, 1, 19);
    test("abcdefghijklmnopqrst", 0, 19, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 0, 0, 19);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 0, 1, 18);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 0, 5, 14);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 0, 9, 10);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 0, 10, 9);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 0, 11, 9);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 1, 0, 19);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 1, 1, -1);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 1, 4, -1);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 1, 8, -1);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 1, 9, -1);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 1, 10, -1);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 5, 0, 19);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 5, 1, -5);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 5, 2, -5);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 5, 4, -5);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 5, 5, -5);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 5, 6, -5);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 9, 0, 19);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 9, 1, -9);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 9, 2, -9);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 10, 0, 19);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 10, 1, 19);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 0, 0, 19);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 0, 1, 18);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 0, 10, 9);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 0, 19, 0);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 0, 20, -1);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 0, 21, -1);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 1, 0, 19);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 1, 9, -1);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 1, 18, -1);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 1, 19, -1);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 1, 20, -1);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 10, 0, 19);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 10, 1, -10);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 10, 5, -10);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 10, 9, -10);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 19, 0, 19);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 19, 1, -19);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 19, 2, -19);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 20, 0, 19);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 20, 1, 19);
    test("abcdefghijklmnopqrst", 0, 19, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 0, 20, "", 0, 0, 20);
    test("abcdefghijklmnopqrst", 0, 20, "", 0, 1, 20);
    test("abcdefghijklmnopqrst", 0, 20, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 0, 0, 20);
}

void test40()
{
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 0, 1, 19);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 0, 2, 18);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 0, 4, 16);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 0, 5, 15);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 0, 6, 15);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 1, 0, 20);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 1, 1, -1);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 1, 2, -1);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 1, 3, -1);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 1, 4, -1);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 1, 5, -1);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 2, 0, 20);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 2, 1, -2);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 2, 2, -2);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 2, 3, -2);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 2, 4, -2);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 4, 0, 20);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 4, 1, -4);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 4, 2, -4);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 5, 0, 20);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 5, 1, 20);
    test("abcdefghijklmnopqrst", 0, 20, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 0, 0, 20);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 0, 1, 19);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 0, 5, 15);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 0, 9, 11);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 0, 10, 10);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 0, 11, 10);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 1, 0, 20);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 1, 1, -1);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 1, 4, -1);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 1, 8, -1);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 1, 9, -1);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 1, 10, -1);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 5, 0, 20);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 5, 1, -5);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 5, 2, -5);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 5, 4, -5);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 5, 5, -5);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 5, 6, -5);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 9, 0, 20);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 9, 1, -9);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 9, 2, -9);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 10, 0, 20);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 10, 1, 20);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 0, 0, 20);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 0, 1, 19);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 0, 10, 10);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 0, 19, 1);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 0, 20, 0);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 0, 21, 0);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 1, 0, 20);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 1, 9, -1);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 1, 18, -1);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 1, 19, -1);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 1, 20, -1);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 10, 0, 20);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 10, 1, -10);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 10, 5, -10);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 10, 9, -10);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 19, 0, 20);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 19, 1, -19);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 19, 2, -19);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 20, 0, 20);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 20, 1, 20);
    test("abcdefghijklmnopqrst", 0, 20, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 0, 21, "", 0, 0, 20);
    test("abcdefghijklmnopqrst", 0, 21, "", 0, 1, 20);
    test("abcdefghijklmnopqrst", 0, 21, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 0, 0, 20);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 0, 1, 19);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 0, 2, 18);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 0, 4, 16);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 0, 5, 15);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 0, 6, 15);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 1, 0, 20);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 1, 1, -1);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 1, 2, -1);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 1, 3, -1);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 1, 4, -1);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 1, 5, -1);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 2, 0, 20);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 2, 1, -2);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 2, 2, -2);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 2, 3, -2);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 2, 4, -2);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 4, 0, 20);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 4, 1, -4);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 4, 2, -4);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 5, 0, 20);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 5, 1, 20);
    test("abcdefghijklmnopqrst", 0, 21, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 0, 0, 20);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 0, 1, 19);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 0, 5, 15);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 0, 9, 11);
}

void test41()
{
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 0, 10, 10);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 0, 11, 10);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 1, 0, 20);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 1, 1, -1);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 1, 4, -1);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 1, 8, -1);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 1, 9, -1);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 1, 10, -1);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 5, 0, 20);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 5, 1, -5);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 5, 2, -5);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 5, 4, -5);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 5, 5, -5);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 5, 6, -5);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 9, 0, 20);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 9, 1, -9);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 9, 2, -9);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 10, 0, 20);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 10, 1, 20);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 0, 0, 20);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 0, 1, 19);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 0, 10, 10);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 0, 19, 1);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 0, 20, 0);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 0, 21, 0);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 1, 0, 20);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 1, 9, -1);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 1, 18, -1);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 1, 19, -1);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 1, 20, -1);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 10, 0, 20);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 10, 1, -10);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 10, 5, -10);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 10, 9, -10);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 19, 0, 20);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 19, 1, -19);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 19, 2, -19);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 20, 0, 20);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 20, 1, 20);
    test("abcdefghijklmnopqrst", 0, 21, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 1, 0, "", 0, 0, 0);
    test("abcdefghijklmnopqrst", 1, 0, "", 0, 1, 0);
    test("abcdefghijklmnopqrst", 1, 0, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 0, 0, 0);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 0, 1, -1);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 0, 2, -2);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 0, 4, -4);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 0, 5, -5);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 0, 6, -5);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 1, 0, 0);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 1, 1, -1);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 1, 2, -2);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 1, 3, -3);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 1, 4, -4);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 1, 5, -4);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 2, 0, 0);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 2, 1, -1);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 2, 2, -2);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 2, 3, -3);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 2, 4, -3);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 4, 0, 0);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 4, 1, -1);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 4, 2, -1);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 5, 0, 0);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 5, 1, 0);
    test("abcdefghijklmnopqrst", 1, 0, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 0, 0, 0);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 0, 1, -1);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 0, 5, -5);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 0, 9, -9);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 0, 10, -10);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 0, 11, -10);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 1, 0, 0);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 1, 1, -1);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 1, 4, -4);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 1, 8, -8);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 1, 9, -9);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 1, 10, -9);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 5, 0, 0);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 5, 1, -1);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 5, 2, -2);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 5, 4, -4);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 5, 5, -5);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 5, 6, -5);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 9, 0, 0);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 9, 1, -1);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 9, 2, -1);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 10, 0, 0);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 10, 1, 0);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 0, 0, 0);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 0, 1, -1);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 0, 10, -10);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 0, 19, -19);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 0, 20, -20);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 0, 21, -20);
}

void test42()
{
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 1, 0, 0);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 1, 9, -9);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 1, 18, -18);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 1, 19, -19);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 1, 20, -19);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 10, 0, 0);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 10, 1, -1);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 19, 0, 0);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 19, 1, -1);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 19, 2, -1);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 20, 0, 0);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 20, 1, 0);
    test("abcdefghijklmnopqrst", 1, 0, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 1, 1, "", 0, 0, 1);
    test("abcdefghijklmnopqrst", 1, 1, "", 0, 1, 1);
    test("abcdefghijklmnopqrst", 1, 1, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 0, 0, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 0, 1, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 0, 2, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 0, 4, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 0, 5, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 0, 6, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 1, 0, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 1, 1, 0);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 1, 2, -1);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 1, 3, -2);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 1, 4, -3);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 1, 5, -3);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 2, 0, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 2, 1, -1);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 2, 2, -1);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 2, 3, -1);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 2, 4, -1);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 4, 0, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 4, 1, -3);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 4, 2, -3);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 5, 0, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 5, 1, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 0, 0, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 0, 1, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 0, 5, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 0, 9, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 0, 10, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 0, 11, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 1, 0, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 1, 1, 0);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 1, 4, -3);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 1, 8, -7);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 1, 9, -8);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 1, 10, -8);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 5, 0, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 5, 1, -4);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 5, 2, -4);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 5, 4, -4);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 5, 5, -4);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 5, 6, -4);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 9, 0, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 9, 1, -8);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 9, 2, -8);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 10, 0, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 10, 1, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 0, 0, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 0, 1, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 0, 10, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 0, 19, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 0, 20, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 0, 21, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 1, 0, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 1, 1, 0);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 1, 9, -8);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 1, 18, -17);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 1, 19, -18);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 1, 20, -18);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 10, 0, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 10, 1, -9);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 10, 5, -9);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 10, 10, -9);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 10, 11, -9);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 19, 0, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 19, 1, -18);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 19, 2, -18);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 20, 0, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 20, 1, 1);
    test("abcdefghijklmnopqrst", 1, 1, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 1, 9, "", 0, 0, 9);
    test("abcdefghijklmnopqrst", 1, 9, "", 0, 1, 9);
    test("abcdefghijklmnopqrst", 1, 9, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 0, 0, 9);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 0, 1, 1);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 0, 2, 1);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 0, 4, 1);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 0, 5, 1);
}

void test43()
{
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 0, 6, 1);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 1, 0, 9);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 1, 1, 8);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 1, 2, 7);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 1, 3, 6);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 1, 4, 5);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 1, 5, 5);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 2, 0, 9);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 2, 1, -1);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 2, 2, -1);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 2, 3, -1);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 2, 4, -1);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 4, 0, 9);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 4, 1, -3);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 4, 2, -3);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 5, 0, 9);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 5, 1, 9);
    test("abcdefghijklmnopqrst", 1, 9, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 0, 0, 9);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 0, 1, 1);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 0, 5, 1);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 0, 9, 1);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 0, 10, 1);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 0, 11, 1);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 1, 0, 9);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 1, 1, 8);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 1, 4, 5);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 1, 8, 1);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 1, 9, 0);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 1, 10, 0);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 5, 0, 9);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 5, 1, -4);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 5, 2, -4);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 5, 4, -4);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 5, 5, -4);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 5, 6, -4);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 9, 0, 9);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 9, 1, -8);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 9, 2, -8);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 10, 0, 9);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 10, 1, 9);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 0, 0, 9);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 0, 1, 1);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 0, 10, 1);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 0, 19, 1);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 0, 20, 1);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 0, 21, 1);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 1, 0, 9);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 1, 1, 8);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 1, 9, 0);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 1, 18, -9);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 1, 19, -10);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 1, 20, -10);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 10, 0, 9);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 10, 1, -9);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 10, 5, -9);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 10, 10, -9);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 10, 11, -9);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 19, 0, 9);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 19, 1, -18);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 19, 2, -18);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 20, 0, 9);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 20, 1, 9);
    test("abcdefghijklmnopqrst", 1, 9, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 1, 18, "", 0, 0, 18);
    test("abcdefghijklmnopqrst", 1, 18, "", 0, 1, 18);
    test("abcdefghijklmnopqrst", 1, 18, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 0, 0, 18);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 0, 1, 1);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 0, 2, 1);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 0, 4, 1);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 0, 5, 1);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 0, 6, 1);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 1, 0, 18);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 1, 1, 17);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 1, 2, 16);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 1, 3, 15);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 1, 4, 14);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 1, 5, 14);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 2, 0, 18);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 2, 1, -1);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 2, 2, -1);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 2, 3, -1);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 2, 4, -1);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 4, 0, 18);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 4, 1, -3);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 4, 2, -3);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 5, 0, 18);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 5, 1, 18);
    test("abcdefghijklmnopqrst", 1, 18, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 0, 0, 18);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 0, 1, 1);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 0, 5, 1);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 0, 9, 1);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 0, 10, 1);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 0, 11, 1);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 1, 0, 18);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 1, 1, 17);
}

void test44()
{
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 1, 4, 14);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 1, 8, 10);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 1, 9, 9);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 1, 10, 9);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 5, 0, 18);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 5, 1, -4);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 5, 2, -4);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 5, 4, -4);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 5, 5, -4);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 5, 6, -4);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 9, 0, 18);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 9, 1, -8);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 9, 2, -8);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 10, 0, 18);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 10, 1, 18);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 0, 0, 18);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 0, 1, 1);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 0, 10, 1);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 0, 19, 1);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 0, 20, 1);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 0, 21, 1);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 1, 0, 18);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 1, 1, 17);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 1, 9, 9);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 1, 18, 0);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 1, 19, -1);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 1, 20, -1);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 10, 0, 18);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 10, 1, -9);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 10, 5, -9);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 10, 10, -9);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 10, 11, -9);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 19, 0, 18);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 19, 1, -18);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 19, 2, -18);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 20, 0, 18);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 20, 1, 18);
    test("abcdefghijklmnopqrst", 1, 18, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 1, 19, "", 0, 0, 19);
    test("abcdefghijklmnopqrst", 1, 19, "", 0, 1, 19);
    test("abcdefghijklmnopqrst", 1, 19, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 0, 0, 19);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 0, 1, 1);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 0, 2, 1);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 0, 4, 1);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 0, 5, 1);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 0, 6, 1);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 1, 0, 19);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 1, 1, 18);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 1, 2, 17);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 1, 3, 16);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 1, 4, 15);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 1, 5, 15);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 2, 0, 19);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 2, 1, -1);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 2, 2, -1);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 2, 3, -1);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 2, 4, -1);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 4, 0, 19);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 4, 1, -3);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 4, 2, -3);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 5, 0, 19);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 5, 1, 19);
    test("abcdefghijklmnopqrst", 1, 19, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 0, 0, 19);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 0, 1, 1);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 0, 5, 1);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 0, 9, 1);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 0, 10, 1);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 0, 11, 1);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 1, 0, 19);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 1, 1, 18);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 1, 4, 15);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 1, 8, 11);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 1, 9, 10);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 1, 10, 10);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 5, 0, 19);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 5, 1, -4);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 5, 2, -4);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 5, 4, -4);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 5, 5, -4);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 5, 6, -4);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 9, 0, 19);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 9, 1, -8);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 9, 2, -8);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 10, 0, 19);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 10, 1, 19);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 0, 0, 19);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 0, 1, 1);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 0, 10, 1);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 0, 19, 1);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 0, 20, 1);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 0, 21, 1);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 1, 0, 19);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 1, 1, 18);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 1, 9, 10);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 1, 18, 1);
}

void test45()
{
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 1, 19, 0);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 1, 20, 0);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 10, 0, 19);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 10, 1, -9);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 10, 5, -9);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 10, 10, -9);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 10, 11, -9);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 19, 0, 19);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 19, 1, -18);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 19, 2, -18);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 20, 0, 19);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 20, 1, 19);
    test("abcdefghijklmnopqrst", 1, 19, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 1, 20, "", 0, 0, 19);
    test("abcdefghijklmnopqrst", 1, 20, "", 0, 1, 19);
    test("abcdefghijklmnopqrst", 1, 20, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 0, 0, 19);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 0, 1, 1);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 0, 2, 1);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 0, 4, 1);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 0, 5, 1);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 0, 6, 1);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 1, 0, 19);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 1, 1, 18);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 1, 2, 17);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 1, 3, 16);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 1, 4, 15);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 1, 5, 15);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 2, 0, 19);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 2, 1, -1);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 2, 2, -1);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 2, 3, -1);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 2, 4, -1);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 4, 0, 19);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 4, 1, -3);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 4, 2, -3);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 5, 0, 19);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 5, 1, 19);
    test("abcdefghijklmnopqrst", 1, 20, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 0, 0, 19);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 0, 1, 1);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 0, 5, 1);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 0, 9, 1);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 0, 10, 1);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 0, 11, 1);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 1, 0, 19);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 1, 1, 18);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 1, 4, 15);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 1, 8, 11);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 1, 9, 10);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 1, 10, 10);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 5, 0, 19);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 5, 1, -4);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 5, 2, -4);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 5, 4, -4);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 5, 5, -4);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 5, 6, -4);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 9, 0, 19);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 9, 1, -8);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 9, 2, -8);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 10, 0, 19);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 10, 1, 19);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 0, 0, 19);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 0, 1, 1);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 0, 10, 1);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 0, 19, 1);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 0, 20, 1);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 0, 21, 1);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 1, 0, 19);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 1, 1, 18);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 1, 9, 10);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 1, 18, 1);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 1, 19, 0);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 1, 20, 0);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 10, 0, 19);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 10, 1, -9);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 10, 5, -9);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 10, 10, -9);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 10, 11, -9);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 19, 0, 19);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 19, 1, -18);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 19, 2, -18);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 20, 0, 19);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 20, 1, 19);
    test("abcdefghijklmnopqrst", 1, 20, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 10, 0, "", 0, 0, 0);
    test("abcdefghijklmnopqrst", 10, 0, "", 0, 1, 0);
    test("abcdefghijklmnopqrst", 10, 0, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 0, 0, 0);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 0, 1, -1);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 0, 2, -2);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 0, 4, -4);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 0, 5, -5);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 0, 6, -5);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 1, 0, 0);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 1, 1, -1);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 1, 2, -2);
}

void test46()
{
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 1, 3, -3);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 1, 4, -4);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 1, 5, -4);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 2, 0, 0);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 2, 1, -1);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 2, 2, -2);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 2, 3, -3);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 2, 4, -3);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 4, 0, 0);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 4, 1, -1);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 4, 2, -1);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 5, 0, 0);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 5, 1, 0);
    test("abcdefghijklmnopqrst", 10, 0, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 0, 0, 0);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 0, 1, -1);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 0, 5, -5);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 0, 9, -9);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 0, 10, -10);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 0, 11, -10);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 1, 0, 0);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 1, 1, -1);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 1, 4, -4);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 1, 8, -8);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 1, 9, -9);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 1, 10, -9);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 5, 0, 0);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 5, 1, -1);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 5, 2, -2);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 5, 4, -4);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 5, 5, -5);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 5, 6, -5);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 9, 0, 0);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 9, 1, -1);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 9, 2, -1);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 10, 0, 0);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 10, 1, 0);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 0, 0, 0);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 0, 1, -1);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 0, 10, -10);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 0, 19, -19);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 0, 20, -20);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 0, 21, -20);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 1, 0, 0);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 1, 9, -9);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 1, 18, -18);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 1, 19, -19);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 1, 20, -19);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 10, 0, 0);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 10, 1, -1);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 19, 0, 0);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 19, 1, -1);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 19, 2, -1);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 20, 0, 0);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 20, 1, 0);
    test("abcdefghijklmnopqrst", 10, 0, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 10, 1, "", 0, 0, 1);
    test("abcdefghijklmnopqrst", 10, 1, "", 0, 1, 1);
    test("abcdefghijklmnopqrst", 10, 1, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 0, 0, 1);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 0, 1, 10);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 0, 2, 10);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 0, 4, 10);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 0, 5, 10);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 0, 6, 10);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 1, 0, 1);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 1, 1, 9);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 1, 2, 9);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 1, 3, 9);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 1, 4, 9);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 1, 5, 9);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 2, 0, 1);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 2, 1, 8);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 2, 2, 8);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 2, 3, 8);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 2, 4, 8);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 4, 0, 1);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 4, 1, 6);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 4, 2, 6);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 5, 0, 1);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 5, 1, 1);
    test("abcdefghijklmnopqrst", 10, 1, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 0, 0, 1);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 0, 1, 10);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 0, 5, 10);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 0, 9, 10);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 0, 10, 10);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 0, 11, 10);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 1, 0, 1);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 1, 1, 9);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 1, 4, 9);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 1, 8, 9);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 1, 9, 9);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 1, 10, 9);
}

void test47()
{
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 5, 0, 1);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 5, 1, 5);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 5, 2, 5);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 5, 4, 5);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 5, 5, 5);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 5, 6, 5);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 9, 0, 1);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 9, 1, 1);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 9, 2, 1);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 10, 0, 1);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 10, 1, 1);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 0, 0, 1);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 0, 1, 10);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 0, 10, 10);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 0, 19, 10);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 0, 20, 10);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 0, 21, 10);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 1, 0, 1);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 1, 1, 9);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 1, 9, 9);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 1, 18, 9);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 1, 19, 9);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 1, 20, 9);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 10, 0, 1);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 10, 1, 0);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 10, 5, -4);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 10, 9, -8);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 10, 10, -9);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 10, 11, -9);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 19, 0, 1);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 19, 1, -9);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 19, 2, -9);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 20, 0, 1);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 20, 1, 1);
    test("abcdefghijklmnopqrst", 10, 1, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 10, 5, "", 0, 0, 5);
    test("abcdefghijklmnopqrst", 10, 5, "", 0, 1, 5);
    test("abcdefghijklmnopqrst", 10, 5, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 0, 0, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 0, 1, 10);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 0, 2, 10);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 0, 4, 10);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 0, 5, 10);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 0, 6, 10);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 1, 0, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 1, 1, 9);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 1, 2, 9);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 1, 3, 9);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 1, 4, 9);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 1, 5, 9);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 2, 0, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 2, 1, 8);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 2, 2, 8);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 2, 3, 8);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 2, 4, 8);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 4, 0, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 4, 1, 6);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 4, 2, 6);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 5, 0, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 5, 1, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 0, 0, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 0, 1, 10);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 0, 5, 10);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 0, 9, 10);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 0, 10, 10);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 0, 11, 10);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 1, 0, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 1, 1, 9);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 1, 4, 9);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 1, 8, 9);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 1, 9, 9);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 1, 10, 9);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 5, 0, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 5, 1, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 5, 2, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 5, 4, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 5, 5, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 5, 6, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 9, 0, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 9, 1, 1);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 9, 2, 1);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 10, 0, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 10, 1, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 0, 0, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 0, 1, 10);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 0, 10, 10);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 0, 19, 10);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 0, 20, 10);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 0, 21, 10);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 1, 0, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 1, 1, 9);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 1, 9, 9);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 1, 18, 9);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 1, 19, 9);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 1, 20, 9);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 10, 0, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 10, 1, 4);
}

void test48()
{
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 10, 5, 0);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 10, 9, -4);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 10, 10, -5);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 10, 11, -5);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 19, 0, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 19, 1, -9);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 19, 2, -9);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 20, 0, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 20, 1, 5);
    test("abcdefghijklmnopqrst", 10, 5, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 10, 9, "", 0, 0, 9);
    test("abcdefghijklmnopqrst", 10, 9, "", 0, 1, 9);
    test("abcdefghijklmnopqrst", 10, 9, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 0, 0, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 0, 1, 10);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 0, 2, 10);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 0, 4, 10);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 0, 5, 10);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 0, 6, 10);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 1, 0, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 1, 1, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 1, 2, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 1, 3, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 1, 4, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 1, 5, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 2, 0, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 2, 1, 8);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 2, 2, 8);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 2, 3, 8);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 2, 4, 8);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 4, 0, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 4, 1, 6);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 4, 2, 6);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 5, 0, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 5, 1, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 0, 0, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 0, 1, 10);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 0, 5, 10);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 0, 9, 10);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 0, 10, 10);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 0, 11, 10);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 1, 0, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 1, 1, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 1, 4, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 1, 8, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 1, 9, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 1, 10, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 5, 0, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 5, 1, 5);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 5, 2, 5);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 5, 4, 5);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 5, 5, 5);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 5, 6, 5);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 9, 0, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 9, 1, 1);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 9, 2, 1);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 10, 0, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 10, 1, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 0, 0, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 0, 1, 10);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 0, 10, 10);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 0, 19, 10);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 0, 20, 10);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 0, 21, 10);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 1, 0, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 1, 1, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 1, 9, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 1, 18, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 1, 19, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 1, 20, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 10, 0, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 10, 1, 8);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 10, 5, 4);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 10, 9, 0);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 10, 10, -1);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 10, 11, -1);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 19, 0, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 19, 1, -9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 19, 2, -9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 20, 0, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 20, 1, 9);
    test("abcdefghijklmnopqrst", 10, 9, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 10, 10, "", 0, 0, 10);
    test("abcdefghijklmnopqrst", 10, 10, "", 0, 1, 10);
    test("abcdefghijklmnopqrst", 10, 10, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 0, 0, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 0, 1, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 0, 2, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 0, 4, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 0, 5, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 0, 6, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 1, 0, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 1, 1, 9);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 1, 2, 9);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 1, 3, 9);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 1, 4, 9);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 1, 5, 9);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 2, 0, 10);
}

void test49()
{
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 2, 1, 8);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 2, 2, 8);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 2, 3, 8);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 2, 4, 8);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 4, 0, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 4, 1, 6);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 4, 2, 6);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 5, 0, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 5, 1, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 0, 0, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 0, 1, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 0, 5, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 0, 9, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 0, 10, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 0, 11, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 1, 0, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 1, 1, 9);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 1, 4, 9);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 1, 8, 9);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 1, 9, 9);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 1, 10, 9);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 5, 0, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 5, 1, 5);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 5, 2, 5);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 5, 4, 5);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 5, 5, 5);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 5, 6, 5);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 9, 0, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 9, 1, 1);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 9, 2, 1);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 10, 0, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 10, 1, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 0, 0, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 0, 1, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 0, 10, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 0, 19, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 0, 20, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 0, 21, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 1, 0, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 1, 1, 9);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 1, 9, 9);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 1, 18, 9);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 1, 19, 9);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 1, 20, 9);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 10, 0, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 10, 1, 9);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 10, 5, 5);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 10, 9, 1);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 10, 10, 0);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 10, 11, 0);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 19, 0, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 19, 1, -9);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 19, 2, -9);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 20, 0, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 20, 1, 10);
    test("abcdefghijklmnopqrst", 10, 10, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 10, 11, "", 0, 0, 10);
    test("abcdefghijklmnopqrst", 10, 11, "", 0, 1, 10);
    test("abcdefghijklmnopqrst", 10, 11, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 0, 0, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 0, 1, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 0, 2, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 0, 4, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 0, 5, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 0, 6, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 1, 0, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 1, 1, 9);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 1, 2, 9);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 1, 3, 9);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 1, 4, 9);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 1, 5, 9);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 2, 0, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 2, 1, 8);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 2, 2, 8);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 2, 3, 8);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 2, 4, 8);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 4, 0, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 4, 1, 6);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 4, 2, 6);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 5, 0, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 5, 1, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 0, 0, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 0, 1, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 0, 5, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 0, 9, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 0, 10, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 0, 11, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 1, 0, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 1, 1, 9);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 1, 4, 9);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 1, 8, 9);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 1, 9, 9);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 1, 10, 9);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 5, 0, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 5, 1, 5);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 5, 2, 5);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 5, 4, 5);
}

void test50()
{
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 5, 5, 5);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 5, 6, 5);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 9, 0, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 9, 1, 1);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 9, 2, 1);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 10, 0, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 10, 1, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 0, 0, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 0, 1, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 0, 10, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 0, 19, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 0, 20, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 0, 21, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 1, 0, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 1, 1, 9);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 1, 9, 9);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 1, 18, 9);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 1, 19, 9);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 1, 20, 9);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 10, 0, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 10, 1, 9);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 10, 5, 5);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 10, 9, 1);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 10, 10, 0);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 10, 11, 0);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 19, 0, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 19, 1, -9);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 19, 2, -9);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 20, 0, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 20, 1, 10);
    test("abcdefghijklmnopqrst", 10, 11, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 19, 0, "", 0, 0, 0);
    test("abcdefghijklmnopqrst", 19, 0, "", 0, 1, 0);
    test("abcdefghijklmnopqrst", 19, 0, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 0, 0, 0);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 0, 1, -1);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 0, 2, -2);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 0, 4, -4);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 0, 5, -5);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 0, 6, -5);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 1, 0, 0);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 1, 1, -1);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 1, 2, -2);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 1, 3, -3);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 1, 4, -4);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 1, 5, -4);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 2, 0, 0);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 2, 1, -1);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 2, 2, -2);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 2, 3, -3);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 2, 4, -3);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 4, 0, 0);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 4, 1, -1);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 4, 2, -1);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 5, 0, 0);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 5, 1, 0);
    test("abcdefghijklmnopqrst", 19, 0, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 0, 0, 0);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 0, 1, -1);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 0, 5, -5);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 0, 9, -9);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 0, 10, -10);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 0, 11, -10);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 1, 0, 0);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 1, 1, -1);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 1, 4, -4);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 1, 8, -8);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 1, 9, -9);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 1, 10, -9);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 5, 0, 0);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 5, 1, -1);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 5, 2, -2);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 5, 4, -4);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 5, 5, -5);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 5, 6, -5);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 9, 0, 0);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 9, 1, -1);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 9, 2, -1);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 10, 0, 0);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 10, 1, 0);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 0, 0, 0);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 0, 1, -1);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 0, 10, -10);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 0, 19, -19);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 0, 20, -20);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 0, 21, -20);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 1, 0, 0);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 1, 9, -9);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 1, 18, -18);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 1, 19, -19);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 1, 20, -19);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 10, 0, 0);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 10, 1, -1);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 10, 11, -10);
}

void test51()
{
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 19, 0, 0);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 19, 1, -1);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 19, 2, -1);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 20, 0, 0);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 20, 1, 0);
    test("abcdefghijklmnopqrst", 19, 0, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 19, 1, "", 0, 0, 1);
    test("abcdefghijklmnopqrst", 19, 1, "", 0, 1, 1);
    test("abcdefghijklmnopqrst", 19, 1, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 0, 0, 1);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 0, 1, 19);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 0, 2, 19);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 0, 4, 19);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 0, 5, 19);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 0, 6, 19);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 1, 0, 1);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 1, 1, 18);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 1, 2, 18);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 1, 3, 18);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 1, 4, 18);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 1, 5, 18);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 2, 0, 1);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 2, 1, 17);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 2, 2, 17);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 2, 3, 17);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 2, 4, 17);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 4, 0, 1);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 4, 1, 15);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 4, 2, 15);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 5, 0, 1);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 5, 1, 1);
    test("abcdefghijklmnopqrst", 19, 1, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 0, 0, 1);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 0, 1, 19);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 0, 5, 19);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 0, 9, 19);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 0, 10, 19);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 0, 11, 19);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 1, 0, 1);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 1, 1, 18);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 1, 4, 18);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 1, 8, 18);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 1, 9, 18);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 1, 10, 18);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 5, 0, 1);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 5, 1, 14);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 5, 2, 14);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 5, 4, 14);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 5, 5, 14);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 5, 6, 14);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 9, 0, 1);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 9, 1, 10);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 9, 2, 10);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 10, 0, 1);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 10, 1, 1);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 0, 0, 1);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 0, 1, 19);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 0, 10, 19);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 0, 19, 19);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 0, 20, 19);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 0, 21, 19);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 1, 0, 1);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 1, 1, 18);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 1, 9, 18);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 1, 18, 18);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 1, 19, 18);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 1, 20, 18);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 10, 0, 1);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 10, 1, 9);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 10, 5, 9);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 10, 9, 9);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 10, 10, 9);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 10, 11, 9);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 19, 0, 1);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 19, 1, 0);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 19, 2, 0);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 20, 0, 1);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 20, 1, 1);
    test("abcdefghijklmnopqrst", 19, 1, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 19, 2, "", 0, 0, 1);
    test("abcdefghijklmnopqrst", 19, 2, "", 0, 1, 1);
    test("abcdefghijklmnopqrst", 19, 2, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 0, 0, 1);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 0, 1, 19);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 0, 2, 19);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 0, 4, 19);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 0, 5, 19);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 0, 6, 19);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 1, 0, 1);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 1, 1, 18);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 1, 2, 18);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 1, 3, 18);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 1, 4, 18);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 1, 5, 18);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 2, 0, 1);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 2, 1, 17);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 2, 2, 17);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 2, 3, 17);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 2, 4, 17);
}

void test52()
{
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 4, 0, 1);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 4, 1, 15);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 4, 2, 15);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 5, 0, 1);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 5, 1, 1);
    test("abcdefghijklmnopqrst", 19, 2, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 0, 0, 1);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 0, 1, 19);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 0, 5, 19);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 0, 9, 19);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 0, 10, 19);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 0, 11, 19);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 1, 0, 1);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 1, 1, 18);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 1, 4, 18);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 1, 8, 18);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 1, 9, 18);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 1, 10, 18);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 5, 0, 1);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 5, 1, 14);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 5, 2, 14);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 5, 4, 14);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 5, 5, 14);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 5, 6, 14);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 9, 0, 1);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 9, 1, 10);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 9, 2, 10);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 10, 0, 1);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 10, 1, 1);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 0, 0, 1);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 0, 1, 19);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 0, 10, 19);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 0, 19, 19);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 0, 20, 19);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 0, 21, 19);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 1, 0, 1);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 1, 1, 18);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 1, 9, 18);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 1, 18, 18);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 1, 19, 18);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 1, 20, 18);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 10, 0, 1);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 10, 1, 9);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 10, 5, 9);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 10, 9, 9);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 10, 10, 9);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 10, 11, 9);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 19, 0, 1);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 19, 1, 0);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 19, 2, 0);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 20, 0, 1);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 20, 1, 1);
    test("abcdefghijklmnopqrst", 19, 2, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 20, 0, "", 0, 0, 0);
    test("abcdefghijklmnopqrst", 20, 0, "", 0, 1, 0);
    test("abcdefghijklmnopqrst", 20, 0, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 0, 0, 0);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 0, 1, -1);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 0, 2, -2);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 0, 4, -4);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 0, 5, -5);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 0, 6, -5);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 1, 0, 0);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 1, 1, -1);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 1, 2, -2);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 1, 3, -3);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 1, 4, -4);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 1, 5, -4);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 2, 0, 0);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 2, 1, -1);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 2, 2, -2);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 2, 3, -3);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 2, 4, -3);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 4, 0, 0);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 4, 1, -1);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 4, 2, -1);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 5, 0, 0);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 5, 1, 0);
    test("abcdefghijklmnopqrst", 20, 0, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 0, 0, 0);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 0, 1, -1);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 0, 5, -5);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 0, 9, -9);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 0, 10, -10);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 0, 11, -10);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 1, 0, 0);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 1, 1, -1);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 1, 4, -4);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 1, 8, -8);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 1, 9, -9);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 1, 10, -9);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 5, 0, 0);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 5, 1, -1);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 5, 2, -2);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 5, 4, -4);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 5, 5, -5);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 5, 6, -5);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 9, 0, 0);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 9, 1, -1);
}

void test53()
{
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 9, 2, -1);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 10, 0, 0);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 10, 1, 0);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 0, 0, 0);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 0, 1, -1);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 0, 10, -10);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 0, 19, -19);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 0, 20, -20);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 0, 21, -20);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 1, 0, 0);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 1, 9, -9);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 1, 18, -18);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 1, 19, -19);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 1, 20, -19);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 10, 0, 0);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 10, 1, -1);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 19, 0, 0);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 19, 1, -1);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 19, 2, -1);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 20, 0, 0);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 20, 1, 0);
    test("abcdefghijklmnopqrst", 20, 0, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 20, 1, "", 0, 0, 0);
    test("abcdefghijklmnopqrst", 20, 1, "", 0, 1, 0);
    test("abcdefghijklmnopqrst", 20, 1, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 0, 0, 0);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 0, 1, -1);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 0, 2, -2);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 0, 4, -4);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 0, 5, -5);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 0, 6, -5);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 1, 0, 0);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 1, 1, -1);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 1, 2, -2);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 1, 3, -3);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 1, 4, -4);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 1, 5, -4);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 2, 0, 0);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 2, 1, -1);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 2, 2, -2);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 2, 3, -3);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 2, 4, -3);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 4, 0, 0);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 4, 1, -1);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 4, 2, -1);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 5, 0, 0);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 5, 1, 0);
    test("abcdefghijklmnopqrst", 20, 1, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 0, 0, 0);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 0, 1, -1);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 0, 5, -5);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 0, 9, -9);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 0, 10, -10);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 0, 11, -10);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 1, 0, 0);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 1, 1, -1);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 1, 4, -4);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 1, 8, -8);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 1, 9, -9);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 1, 10, -9);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 5, 0, 0);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 5, 1, -1);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 5, 2, -2);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 5, 4, -4);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 5, 5, -5);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 5, 6, -5);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 9, 0, 0);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 9, 1, -1);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 9, 2, -1);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 10, 0, 0);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 10, 1, 0);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 0, 0, 0);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 0, 1, -1);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 0, 10, -10);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 0, 19, -19);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 0, 20, -20);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 0, 21, -20);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 1, 0, 0);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 1, 1, -1);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 1, 9, -9);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 1, 18, -18);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 1, 19, -19);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 1, 20, -19);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 10, 0, 0);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 10, 1, -1);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 10, 5, -5);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 10, 9, -9);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 10, 10, -10);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 10, 11, -10);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 19, 0, 0);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 19, 1, -1);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 19, 2, -1);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 20, 0, 0);
}

void test54()
{
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 20, 1, 0);
    test("abcdefghijklmnopqrst", 20, 1, "abcdefghijklmnopqrst", 21, 0, 0);
    test("abcdefghijklmnopqrst", 21, 0, "", 0, 0, 0);
    test("abcdefghijklmnopqrst", 21, 0, "", 0, 1, 0);
    test("abcdefghijklmnopqrst", 21, 0, "", 1, 0, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 0, 0, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 0, 1, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 0, 2, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 0, 4, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 0, 5, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 0, 6, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 1, 0, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 1, 1, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 1, 2, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 1, 3, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 1, 4, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 1, 5, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 2, 0, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 2, 1, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 2, 2, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 2, 3, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 2, 4, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 4, 0, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 4, 1, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 4, 2, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 5, 0, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 5, 1, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcde", 6, 0, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 0, 0, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 0, 1, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 0, 5, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 0, 9, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 0, 10, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 0, 11, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 1, 0, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 1, 1, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 1, 4, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 1, 8, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 1, 9, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 1, 10, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 5, 0, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 5, 1, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 5, 2, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 5, 4, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 5, 5, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 5, 6, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 9, 0, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 9, 1, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 9, 2, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 10, 0, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 10, 1, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghij", 11, 0, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 0, 0, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 0, 1, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 0, 10, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 0, 19, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 0, 20, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 0, 21, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 1, 0, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 1, 1, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 1, 9, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 1, 18, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 1, 19, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 1, 20, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 10, 0, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 10, 1, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 10, 5, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 10, 9, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 10, 10, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 10, 11, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 19, 0, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 19, 1, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 19, 2, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 20, 0, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 20, 1, 0);
    test("abcdefghijklmnopqrst", 21, 0, "abcdefghijklmnopqrst", 21, 0, 0);
}


int main () {
    test0();
    test1();
    test2();
    test3();
    test4();
    test5();
    test6();
    test7();
    test8();
    test9();
    test10();
    test11();
    test12();
    test13();
    test14();
    test15();
    test16();
    test17();
    test18();
    test19();
    test20();
    test21();
    test22();
    test23();
    test24();
    test25();
    test26();
    test27();
    test28();
    test29();
    test30();
    test31();
    test32();
    test33();
    test34();
    test35();
    test36();
    test37();
    test38();
    test39();
    test40();
    test41();
    test42();
    test43();
    test44();
    test45();
    test46();
    test47();
    test48();
    test49();
    test50();
    test51();
    test52();
    test53();
    test54();


    {
    test("abcde", 5, 1, "", 0, 0, 0);
    test("abcde", 2, 4, "", 0, 0, 3);
    test("abcde", 2, 4, "abcde", 3, 4, -2);
    test("ABCde", 2, 4, "abcde", 2, 4, -1);
    }

    {
    test(L"abcde", 5, 1, L"", 0, 0, 0);
    test(L"abcde", 2, 4, L"", 0, 0, 3);
    test(L"abcde", 2, 4, L"abcde", 3, 4, -2);
    test(L"ABCde", 2, 4, L"abcde", 2, 4, -1);
    }

#if TEST_STD_VER >= 11
    {
    test(u"abcde", 5, 1, u"", 0, 0, 0);
    test(u"abcde", 2, 4, u"", 0, 0, 3);
    test(u"abcde", 2, 4, u"abcde", 3, 4, -2);
    test(u"ABCde", 2, 4, u"abcde", 2, 4, -1);
    }

    {
    test(U"abcde", 5, 1, U"", 0, 0, 0);
    test(U"abcde", 2, 4, U"", 0, 0, 3);
    test(U"abcde", 2, 4, U"abcde", 3, 4, -2);
    test(U"ABCde", 2, 4, U"abcde", 2, 4, -1);
    }
#endif

#if TEST_STD_VER > 11
    {
    typedef std::basic_string_view<char, constexpr_char_traits<char>> SV;
    constexpr SV  sv1 { "abcde", 5 };
    static_assert ( sv1.compare(5, 1, "", 0, 0) == 0, "" );
    static_assert ( sv1.compare(2, 4, "", 0, 0) == 1, "" );
    static_assert ( sv1.compare(2, 4, "abcde", 3, 4) == -1, "" );
    }
#endif
}
